*****************************************************************
*****************************************************************
**
*+ Name: PONG - The game of pong
**
** Author: Douglas R. Cannon
**
** Abstract:   Compatible with all current ROM versions of the HP48
**             as of June 3, 1995.  S, SX, G, or GX.
**
**             Two players, each has a paddle.  The ball will
**             bounce around the screen, each player tries to move
**             his/her paddle in the way of the ball.  The first
**             one to miss the ball loses.
**
**             Keys:
**             [A]  : move left player's paddle up
**             [G]  : move left player's paddle down
**             [F]  : move right player's paddle up
**             [L]  : move right player's paddle down
**             [<-] : EXIT
**             [ON] : EXIT
**
**
** Entry:      None
**
** Exit:       None
**
**     Date     Prog             Modification
**   --------   ----   ------------------------------------------
**   05/06/94   DC     Standard Header
**===============================================================


ASSEMBLE
        NIBASC  /HPHP48-L/

RPL

::

  RECLAIMDISP
  TURNMENUOFF

  ABUFF
  
  #E
  ONE
  MAKEGROB
  DO>STR                        

  56 TestSysFlag                
  
CODE

*****************************************************************
*
* Global name abbr.
*
* db: draw ball
* dl: delay routnies
* ky: key detection routines
* pd: paddle routines
* sc: screen draw
* so: sound routines
* sr: storage/retrieval
*
*
*****************************************************************
*
*  STORAGE VARS
*
*  0 : ptr to ABUFF
*  1 : y coord of left paddle
*  2 : height of left paddle
*  3 : y coord of right paddle
*  4 : height of right paddle
*  5 : screen top
*  6 : screen bottom
*  7 : X of ball
*  8 : Y of ball
*  9 : DIR of ball
*  10: Ball speed
*  11: Old DIR (used in dbLEFT and dbRIGHT)
*  12: Old Y coord (used in dbLEFT and dbRIGHT)
*  13: Old X coord (used in dbDRAW)
*
*****************************************************************
*
*  STATUS FLAGS
*
*  0: UP FLAG
*  1: DOWN FLAG
*  2: LEFT FLAG
*  3: RIGHT FLAG
*  4: Used in dbERLNE
*  5: 0 = move ball again; 1 = don't move ball yet.
*  6: 1 = paddle being moved; 0 = no paddles being moved
*  7:
*  8:
*  9:
* 10:
* 11: Sound on/off
*
*****************************************************************
* 
* Scratch Regs
*
* R0: srSTORAGE addr
* R1:
* R2: dlDELAY
* R3: dlDELAY
* R4: dlDELAY
*
*****************************************************************
*
* When PONG starts, the following things must
* be on the stack:
*
* 3: ABUFF
* 2: String of at least (14 * 5) nibbles storage area
* 1: Flag (TRUE=Sound off / False=Sound on)
*
*****************************************************************
*
* The first time any global ROM routine is called, I have put its 
* address in parenthesis in the comments.  All ROM routines used
* are supported.
*
*****************************************************************



        ST=0    15              Ignore all interrupts
        INTOFF                  Turn off keyboard interrupts
        P=      0
        
        ST=0    11              Assume sound is on
        GOSBVL  =popflag        (#61A02h) pop TRUE or FALSE, 
                                set carry if TRUE
        GONC    DATA
        ST=1    11              Sound is off
  
DATA    GOSBVL  =SAVPTR         (#0679Bh) save RPL pointers
        A=DAT1  A               Read pointer to data string
        A=A+CON A,10            skip past prologue, length
        R0=A.F  A               Store address here forever

        D1=D1+  5               Point to ABUFF
        A=DAT1  A               Read ABUFF pointer
        LC(5)   20
        C=C+A   A               Point past prolog, length, dims
        GOSUB   srS0            Store pointer to ABUFF bitmap here
  
*****************************************************************
*
* Game begin
*
*****************************************************************
*
* Pong was designed to be configurable.  The game itself is very simple,
* and actually quite trivial to play.  However, this code provides a
* good game "shell" that you can manipulate for practice.  If you wish,
* you can build a more powerful pong game from it.  You may also use this
* code for the basis of a different game.  If you do create a better pong
* game from this code, then please post it for others to play!  I would
* also be personally interested in a copy.  You can send it to me at
* dougc@erudite.com.
*
* Each of the variables below can be modified to change the game.
* You can "hard-code" any changes here, or you can build certain logic
* into the game.  For example, if two players volley back and forth 5
* times without missing the ball, you may want to speed the ball up a little.
* If that isn't enough, you may want to start making the paddles shorter!
*
* Currently, the game ends when one player misses the ball.  You can have
* the game go to 10 points, and if one player misses the ball, then have
* the ball start moving in the direction of the victor first.  If someone
* wins 3 times in a row, then you can make his/her paddle shorter for a
* handicap.  The possibilities are endless!
*
*****************************************************************


        C=0     A               This is the Y coordinate of the top line
        GOSUB   srS5            Change this to LC(5) 10 for example, and
                                the top line will be down 10 pixels.  Now
                                you have room for scores, etc. on the top.
                                Because the ball and the paddles use a
                                collision detection, the game automatically
                                works with this change.  Try it!
        
        LC(5)   63              This is the Y coordinate for the bottom
        GOSUB   srS6            line.  This can also be modified.

        LC(5)   24              This is Y coordinate for each paddle.
        GOSUB   srS1            This code shows them at the same location,
        GOSUB   srS3            but this is not necessary.  You can start
                                each paddle wherever you want it.  Be sure
                                that the Y coordinate for each paddle
                                is LARGER than the Y coordinate of the top
                                line.
        
        LC(5)   14              This is the height for each paddle.
        GOSUB   srS2            This code shows them with the same height,
        GOSUB   srS4            but this is not necessary.  If you want to
                                give your opponent a handicap, then make
                                his/her paddle shorter.  It can be as short
                                as 1 pixel, and as long as the distance
                                between the top and bottom lines.  Be sure
                                that the Y coord + height is SMALLER than
                                the Y coordinate of the bottom line.

        LC(5)   48              This is the X coordinate of the ball.
        GOSUB   srS7            You can make it start anywhere on the screen.
        
        LC(5)   8               This is the Y coordinate of the ball.
        GOSUB   srS8

        LCHEX   00006           This is the starting direction that the
        GOSUB   srS9            ball will travel.  In this case, the ball
        GOSUB   srS11           will always start in the same place, going
                                the same direction.  Very boring.  Please
                                add code to make this more interesting!

                                The direction is stored in sr9 and sr11.
                                This should always be the case, sr11 is
                                used in dbLEFT and dbRIGHT.

                                The ball is capable of traveling
                                straight up, straight down, straight left
                                or right, but this makes the game even more
                                boring.  You will want to be sure that
                                the ball is always going diagonally.

                                Bit 0 : Up
                                Bit 1 : Down
                                Bit 2 : Left
                                Bit 3 : Right

                                Therefore:  00005: Up-Left
                                            00009: Up-Right
                                            00006: Down-Left
                                            0000A: Down-Right

        LC(5)   250             Ball delay.  Make this number larger
        GOSUB   srS10           For a slower game, smaller for a faster
                                game.  Note that the ball speed does not
                                affect the speed of the paddles.  They
                                move at the same speed regardless.
                                However, if the ball is moving fast enough,
                                then you will see it slow down each time
                                a paddle is being moved.  (Thus the ball
                                should never be set faster than the paddle
                                delay amount + time it takes to execute
                                one key loop + time to draw the ball once. 
                                about 250 if the paddle delay is #80h; I
                                haven't actually calculated it. )

        GOSUB   scSCRN          Draw the screen (top and bottom line)
        GOSUB   pdLDRAW         Draw the left paddle
        GOSUB   pdRDRAW         Draw the right paddle

        ST=0    0               Clear status flags
        ST=0    1
        ST=0    2
        ST=0    3
        ST=0    5
        ST=0    6
        GOSUB   dbDRAW          Draw the ball


*****************************************************************

kySTART GOSUBL  dlINIT          Initialize the delay routine
kyKEY   LCHEX   5FF             OUT mask for all keys (including ON)
        OUT=C
        GOSBVL  =CINRTN         (#01160h)
        LAHEX   0803F           IN mask for all keys (including ON)
        A=A&C   A
        ?A#0    A               Is there a key being pressed?
        GOYES   kyA             Go check
        GOTO    kyBALL          Otherwise, go move the ball

kyA     LCHEX   002             Check for the A key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00010
        A=A&C   A
        ?A=0    A               A is not being pressed?
        GOYES   kyG             Go check G.

        GOSUB   pdLUP           A is pressed, so move left paddle up one.
        ST=1    6               Signal to do "paddle delay"
        GOTO    kyF             Go check for the F key

kyG     LCHEX   004             Check for the G key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00010
        A=A&C   A
        ?A=0    A               G is not being pressed?
        GOYES   kyF             Go check F.

        GOSUB   pdLDOWN         G is pressed, move left paddle down one.
        ST=1    6               signal to do "paddle delay"

kyF     LCHEX   100             Check for the F key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00001
        A=A&C   A
        ?A=0    A               F is not being pressed?
        GOYES   kyL             Go check L.

        GOSUB   pdRUP           F is pressed, move right paddle up one.
        ST=1    6               signal to do "paddle delay"
        GOTO    kyBALL          Go do the ball code.

kyL     LCHEX   080             Check for the L key.
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00001
        A=A&C   A
        ?A=0    A               L is not being pressed?
        GOYES   kyBS            Go check backspace.

        GOSUB   pdRDOWN         L is pressed, move right paddle down one.
        ST=1    6               signal to do "paddle delay"
        GOTO    kyBALL          Go do the ball code.

kyBS    LCHEX   410             Check for ON or backspace.
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   08001
        A=A&C   A
        ?A=0    A               neither being pressed?
        GOYES   kyBALL          Go do ball code

kyEXIT  LCHEX   410             One is being pressed, wait here
        OUT=C                   until they stop pressing it.
        GOSBVL  =CINRTN
        LAHEX   08001
        A=A&C   A
        ?A#0    A               Still pressing?
        GOYES   kyEXIT          Go wait some more.

        GOVLNG  =GETPTRLOOP     (#05143h) Restore RPL pointers, exit to RPL.


kyBALL  GOSUBL  dlCHECK         Check if the ball delay has expired
        ?ST=0   5               Can we move the ball?
        GOYES   kyYES           Yes, go move the ball.
        ?ST=0   6               are no paddled being moved?
        GOYES   kyDSKIP         skip this delay
        ST=0    6               Clear the paddle flag
        C=0     W
        LCHEX   80              Paddle Delay ( #80h ticks )
        GOSUBL  dlDELAY         Go delay.
                                If there was no delay between paddle
                                moves, then each paddle would move
                                VERY fast, and they really couldn't
                                be controlled.  This value here can
                                be safely modified.  Set it to 01 to
                                see what I mean about the paddles
                                moving VERY fast.  If either (or both)
                                paddles are being moved, then this delay
                                is inserted to make them controlable.

kyDSKIP GOTO    kyKEY           Go to the top of the key loop.

kyYES   GOSUB   kyCDIR          Get ball direction (into ST[0] - ST[3])
        ?ST=0   0               Is the ball not going up?
        GOYES   kyCD            Go check down.
        GOSUBL  dbUP            Move ball up one pixel.
        GOTO    kyCL            Go check Left.  (can't move up AND down).

kyCD    ?ST=0   1               Is the ball not going down?
        GOYES   kyCL            Go check left.
        GOSUBL  dbDOWN          Move ball down one pixel.

kyCL    ?ST=0   2               Is the ball not going left?
        GOYES   kyCR            Go check right.
        GOSUBL  dbLEFT          Move ball left one pixel.
        GOTO    kyMOVE          Go actually draw the ball.

kyCR    ?ST=0   3               Is the ball not going right?
        GOYES   kyMOVE          Go actually draw the ball.
        GOSUBL  dbRIGHT         Move ball right one pixel.

kyMOVE  GOSUB   kyCDIR          Get ball direction into ST[0] - ST[3].
        GOSUB   dbDRAW          Go draw the ball.
        GOTO    kySTART         Go to the very top of the key loop.
                                (The ball timer is reset).

kyCDIR  ST=0    0               UP
        ST=0    1               DOWN
        ST=0    2               LEFT
        ST=0    3               RIGHT ... all cleared
        GOSUB   srR9            Get current ball direction into C[A]
        ?CBIT=0 0               not going up?
        GOYES   kyB1            Go to next
        ST=1    0               Set UP flag
kyB1    ?CBIT=0 1               not going down?
        GOYES   kyB2            go to next
        ST=1    1               Set DOWN flag
kyB2    ?CBIT=0 2               not going left?
        GOYES   kyB3            go to next
        ST=1    2               Set LEFT flag
kyB3    ?CBIT=0 3               not going right?
        RTNYES                  return
        ST=1    3               Set RIGHT flag
        RTN                     return

*****************************************************************
*
* STORAGE SYSTEM
*
* This system has evolved each time I've used it.  It's not a bad system.
* Basically, you have a string on the stack to store the variables into. 
* Any area of RAM will work, but a string on the stack is the safest, and
* if you remove the 2DROP command at the end of the program, you still have
* your storage string on the stack when you exit the game.  This is a 
* GREAT debugging tool.  You have all your variables sitting right there 
* for your inspection.
*
* Anyhow, you can see that the system is very simple.  Each variable is
* numbered, and 5 nibbles in length.  I have rarely (if ever) needed to
* store any variables larger than 5 nibbles in length.  If you absolutely
* MUST do it, then you can use more than one storage location.
*
* srS0 will store C[A] into variable #0.  srR0 will recall variable #0 in
* C[A].  I've made the system so that all registers are preserved so that
* you can store and recall vars without worrying about the other regs getting
* messed up.
* 
* In reality, Recalling a variable preserves all registers, except for C[A],
* of course... that's where the variable is retrieved into!
*
* Storing a variable, modifies A[W], but preserved A[A].  So, if you need
* data in the A reg outside of the A field, then you need to save it first,
* or modify the srSC routine to your liking.
*
* If you need more variables than 0 - 10, just increase the size of the
* data string and add appropriate code for recalling and storing the
* variables.  It's that easy!
*
*****************************************************************

srR0    LC(5)    0
        GOTO     srRC
srR1    LC(5)    1
        GOTO     srRC
srR2    LC(5)    2
        GOTO     srRC
srR3    LC(5)    3
        GOTO     srRC
srR4    LC(5)    4
        GOTO     srRC
srR5    LC(5)    5
        GOTO     srRC
srR6    LC(5)    6
        GOTO     srRC
srR7    LC(5)    7
        GOTO     srRC
srR8    LC(5)    8
        GOTO     srRC
srR9    LC(5)    9
        GOTO     srRC
srR10   LC(5)    10
        GOTO     srRC
srR11   LC(5)    11
        GOTO     srRC
srR12   LC(5)    12
        GOTO     srRC
srR13   LC(5)    13


*****************************************************************
*
* When srRC is called, C[A] must contain the variable number that you
* would like to retrieve.  All regs are preserved even though A[A] and
* D0 are used.  One level of RSTK is used.
*
*****************************************************************

srRC    ACEX     A              Swap A and C
        CD0EX                   Swap D0 and C
        RSTK=C                  Put the orig D0 value onto RSTK
        C=A      A              Copy var # into C.
        C=C+C    A              | Multiply C[A] by 5.
        C=C+C    A              | This is because each var is 5 nibbles
        C=C+A    A              | in length.  This gets the correct offset.
        A=R0.F   A              Get address of data string.
        A=A+C    A              Add the offset to the var #.
        AD0EX                   Restore A[A] (it was in D0).
                                At the same time, this puts the correct
                                Var # RAM address into D0 for use.
        C=DAT0   A              Read the 5 nibble var into C[A].
        CD0EX                   Put the value into D0
        C=RSTK                  Get orig D0 value from RSTK
        CD0EX                   Restore orig D0 value, get var into C[A].
        RTN


srS0    RSTK=C
        LC(5)    0
        GOTO     srSC
srS1    RSTK=C
        LC(5)    1
        GOTO     srSC
srS2    RSTK=C
        LC(5)    2
        GOTO     srSC
srS3    RSTK=C
        LC(5)    3
        GOTO     srSC
srS4    RSTK=C
        LC(5)    4
        GOTO     srSC
srS5    RSTK=C
        LC(5)    5
        GOTO     srSC
srS6    RSTK=C
        LC(5)    6
        GOTO     srSC
srS7    RSTK=C
        LC(5)    7
        GOTO     srSC
srS8    RSTK=C
        LC(5)    8
        GOTO     srSC
srS9    RSTK=C
        LC(5)    9
        GOTO     srSC
srS10   RSTK=C
        LC(5)    10
        GOTO     srSC
srS11   RSTK=C
        LC(5)    11
        GOTO     srSC
srS12   RSTK=C
        LC(5)    12
        GOTO     srSC
srS13   RSTK=C
        LC(5)    13


*****************************************************************
*
* When srSC is called, C[A] must contain the variable number that you
* would like to store, and RSTK should contain the value that will be
* put into it.  All regs are preserved except for A[W].  A[W] is used,
* but A[A] is preserved.  One level of RSTK are used.  (Actually,
* srSC doesn't use an RSTK level, but the calling routine above does.
*
*****************************************************************

srSC    GOSBVL   =ASLW5         (#0D5F6h) Shift A[W] left 5 nibbles 
                                (preserve A[A])
        AD0EX                   Get D0 into A[A]
        GOSBVL   =ASLW5         Shift A[W], preserve D0.
        A=C      A              Get var #
        C=C+C    A              | Multiply C[A] by 5.
        C=C+C    A              | This is because each var is 5 nibbles
        C=C+A    A              | in length.  This gets the correct offset.
        A=R0.F   A              Get address of data string.
        A=A+C    A              Add offset to the var #.
        D0=A                    Put into D0.
        C=RSTK                  Get value off RSTK
        DAT0=C   A              Store value into Variable area.
        GOSBVL   =ASRW5         (#0D5E5h) Put orig D0 into A[A]
        D0=A                    Restore D0
        GOSBVL   =ASRW5         Restore A[A].
        RTN


*****************************************************************
*
* General subroutines
*
*****************************************************************
*
* scSCRN draws the playing field
*
*****************************************************************

scSCRN  GOSUB   srR5            Top line Y coord (into C[A])
        LA(5)   34              34 nibbles per horizontal line
        GOSBVL  =MUL#           (#03991) B[A] = A[A] * C[A]
        GOSUB   srR0            C[A] = ABUFF
        C=C+B   A               Add offset to ABUFF addr
        D0=C                    Put this into D0

        GOSUB   srR6            Bottom line Y coord (into C[A])
        LA(5)   34              34 nibbles per horizontal line
        GOSBVL  =MUL#           B[A] = A[A] * C[A]
        GOSUB   srR0            C[A] = ABUFF
        C=C+B   A               Add offset to ABUFF addr
        D1=C                    Put this into D1

        C=0     W
        C=C-1   W               Put 16 F's into C[W] (a long line!)
        DAT0=C  W               write out 64 pixels of top line
        DAT1=C  W               write 64 pixels of bottom line
        D0=D0+  16
        D1=D1+  16
        DAT0=C  W               write 64 more pixels of top line
        DAT1=C  W               ditto for bottom line
        D0=D0+  16
        D1=D1+  16
        DAT0=C  B               write out 8 more pixles of top line
        DAT1=C  B               ditto for bottom line
                                
                                Even though it looks like I drew a
                                line that was 136 pixels long... the
                                HP48 will only display 131 pixels.
                                (But there's enough RAM for 136!)

*****************************************************************
*
* Because the ball uses collision detection, you can place obstacles
* on the screen and the ball will bounce off of them automatically.
* below are two examples that you can try.  This code is left commented
* out, but you can un-comment it and assemble it to try these
* examples.  It will draw two boxes on the screen that the ball will
* collide with.  
*
*
*        GOSUB   srR0            Get ABUFF
*        LA(5)   692             over 48, down 20.
*        C=C+A   A
*        D0=C
*        LCHEX   FF              8 pixels wide
*        DAT0=C  B
*        GOSUB   D0DOWN
*        DAT0=C  B
*        GOSUB   D0DOWN
*        DAT0=C  B
*        GOSUB   D0DOWN
*        DAT0=C  B               4 pixels tall.
*
*        CD0EX                   Get D0
*        LA(5)   590             over total of 96, down total of 40.
*        C=C+A   A
*        D0=C
*        LCHEX   FF
*        DAT0=C  B
*        GOSUB   D0DOWN
*        DAT0=C  B
*        GOSUB   D0DOWN
*        DAT0=C  B
*        GOSUB   D0DOWN
*        DAT0=C  B               4 pixels tall.
*
*
*****************************************************************

        RTN


*****************************************************************
*
* ADD #22h to D0 (down 1 line)
* This handy routine just moves D0 "down one line" so that if I'm
* drawing something on one line, I can move D0 down to draw something
* just underneath it.  (like when I draw paddles, or the ball, etc.)
*
*****************************************************************

D0DOWN  D0=D0+  16
        D0=D0+  16
        D0=D0+  2
        RTN


*****************************************************************
*
* pdLDRAW draws the left paddle.
* It handles any necessary erasing.
* Also, if the ball is right next to the paddle, it doesn't erase it.
*
*****************************************************************

pdLDRAW GOSUB   srR1            Get Y coord of left paddle
        C=C-1   A               Go up one pixel
        LA(5)   34
        GOSBVL  =MUL#           B[A] = vertical offset
        GOSUB   srR0            Get ABUFF address
        C=C+B   A               Add vertical offset
        D0=C                    D0 gets this address.
                                D0 now points to the nibble address of
                                the screen area just above the left paddle.

        LCHEX   03              Mask for left 2 bits of nibble
        A=DAT0  1               Read 1 nibble from screen
        ?ST=1   1               Is the paddle going down?
        GOYES   pdLERD          Yes.  Erase a line on top regardless.
        C=C&A   B               Mask out 2 screen bits.
        ?C#0    B               If C<>0 then we know that there is a pixel
                                on the screen just above the paddle.
                                we don't want to erase this, it's probably
                                the top line.
        GOYES   pdLTOP          Go draw the paddle

pdLERD  ST=0    1               Reset the DOWN flag.
        LCHEX   C               mask
        C=C&A   B               erase only 2 bits.
        DAT0=C  1               write it out to screen.

pdLTOP  GOSUB   D0DOWN          Move D0 down 1 line.
        GOSUB   srR2            Get height of paddle
        B=C     A
pdLNEXT ?B=0    A               Done drawing?
        GOYES   pdLBOT
        LCHEX   C               mask
        A=DAT0  1               Get screen of this area
        C=C&A   B               Mask screen we want to keep.
        C=C+CON B,3             Add 3 (or 2 pixels on left of nibble)
        DAT0=C  1               Write it out to screen.
        GOSUB   D0DOWN          Go down 1 line.
        B=B-1   A               one less line to draw
        GOTO    pdLNEXT         Go do more.

pdLBOT  LCHEX   03              mask
        A=DAT0  1               Read one screen nibble
        ?ST=1   0               going up?
        GOYES   pdLERU          Erase bottom line regardless.
        C=C&A   B               Mask out 2 screen bits.
        ?C#0    B               If C<>0 then there is a pixel just below
                                the paddle.  We don't want to erase it,
                                it's probably the bottom line on screen.
        GOYES   pdLRTN

pdLERU  ST=0    0               Clear UP flag.
        LCHEX   C               Mask
        C=C&A   B               erase only 2 bits
        DAT0=C  1               write to screen.

pdLRTN  RTN


*****************************************************************
*
* Move the left paddle up one pixel (if nothing is blocking it).
*
*****************************************************************

pdLUP   GOSUB   srR1            Get Y coord of left paddle
        C=C-1   A               go up one
        D=C     A               save it here for a sec.
        LA(5)   34
        GOSBVL  =MUL#           B[A] is vertical offset.
        GOSUB   srR0            Get ABUFF address
        C=C+B   A               Add offset.
        D0=C

        LCHEX   03              mask
        A=DAT0  1               read one screen nibble
        C=C&A   B               Mask off 2 nibbles where the paddle will go.
        ?C#0    B               If C<>0 then that means something is
                                blocking the paddle from going up.
        RTNYES                  In that case, return, we can't go up.

        C=D     A               Otherwise, get new Y coordinate.
        GOSUB   srS1            Store it.
        ST=1    0               Set the moving up flag.
        GOTO    pdLDRAW         Go draw the left paddle.

*****************************************************************
*
* Move the left paddle down one pixel (if nothing is blocking it).
*
*****************************************************************

pdLDOWN GOSUB   srR1            Get the Y coord.
        D=C     A               Store it here for a sec.
        GOSUB   srR2            Get the height.
        C=C+D   A               Add it
        LA(5)   34
        GOSBVL  =MUL#           vertical offset into B[A]
        GOSUB   srR0            ABUFF etc.
        C=C+B   A               Add offset
        D0=C
        
        LCHEX   03              mask
        A=DAT0  1               read 1 screen nibble (just below paddle)
        C=C&A   B
        ?C#0    B               if C<>0 then something is blocking
                                the paddle on the bottom.
        RTNYES                  In that case, we can't move down. RETURN.

        C=D     A               Otherwise, get Y coord
        C=C+1   A               Add one (move down one pixel)
        GOSUB   srS1            Store new value.
        ST=1    1               Set the moving down flag.
        GOTO    pdLDRAW         Go draw the left paddle.



*****************************************************************
*
* pdRDRAW draws the right paddle.
* It handles any necessary erasing.
* Also, if the ball is right next to the paddle, it doesn't erase it.
*
* Many of the comments here would be the same as for pdLDRAW.
* If a comment is missing, refer to pdLDRAW.
*
*****************************************************************

pdRDRAW GOSUB   srR3            Get Y coord of right paddle
        C=C-1   A
        LA(5)   34
        GOSBVL  =MUL#
        GOSUB   srR0
        C=C+B   A
        LA(5)   32
        C=C+A   A               Move to the right side of the screen.
        D0=C

        LCHEX   06              mask
        A=DAT0  1
        ?ST=1   1               Is the paddle going down?
        GOYES   pdRERD
        C=C&A   B
        ?C#0    B
        GOYES   pdRTOP

pdRERD  ST=0    1
        LCHEX   1
        A=DAT0  1
        C=C&A   B
        DAT0=C  1

pdRTOP  GOSUB   D0DOWN
        GOSUB   srR4            Get height of paddle
        B=C     A
pdRNEXT ?B=0    A
        GOYES   pdRBOT
        LCHEX   1
        A=DAT0  1
        C=C&A   B
        C=C+CON B,6
        DAT0=C  1
        GOSUB   D0DOWN
        B=B-1   A
        GOTO    pdRNEXT

pdRBOT  LCHEX   06
        A=DAT0  1
        ?ST=1   0
        GOYES   pdRERU
        C=C&A   B
        ?C#0    B
        GOYES   pdRRTN

pdRERU  ST=0    0
        LCHEX   1
        A=DAT0  1
        C=C&A   B
        DAT0=C  1

pdRRTN  RTN

*****************************************************************
*
* Move the right paddle up one pixel (if nothing is blocking it).
*
*****************************************************************

pdRUP   GOSUB   srR3            Get Y coord of right paddle.
        C=C-1   A
        D=C     A
        LA(5)   34
        GOSBVL  =MUL#
        GOSUB   srR0
        C=C+B   A
        LA(5)   32              Move to the right side of the screen.
        C=C+A   A
        D0=C
        
        LCHEX   06
        A=DAT0  1
        C=C&A   B
        ?C#0    B
        RTNYES
        
        C=D     A
        GOSUB   srS3
        ST=1    0
        GOTO    pdRDRAW


*****************************************************************
*
* Move the right paddle down one pixel (If nothing is blocking it).
*
*****************************************************************

pdRDOWN GOSUB   srR3            Get Y coord of right paddle.
        D=C     A
        GOSUB   srR4
        C=C+D   A
        LA(5)   34
        GOSBVL  =MUL#
        GOSUB   srR0
        C=C+B   A
        LA(5)   32              Move to the right side of the screen.
        C=C+A   A
        D0=C

        LCHEX   06
        A=DAT0  1
        C=C&A   B
        ?C#0    B
        RTNYES

        C=D     A
        C=C+1   A
        GOSUB   srS3
        ST=1    1
        GOTO    pdRDRAW

*****************************************************************
*
* dbDRAW will draw the ball.
*
* Keep in mind that the (X,Y) coordinates of the ball at this point
* have already been modified.  The X coordinate in sr7 is one pixel
* left or right of the ball (depending on direction) and the Y
* coordinate in sr8 is one pixel up or down from the ball (depending
* on direction).  When dbDRAW completes, the ball is in the correct
* position.
*
*
*****************************************************************

dbDRAW  GOSUB   srR8            Y of ball
        C=C-1   A
        LA(5)   34
        GOSBVL  =MUL#
        GOSUB   srR7            X of ball
        C=C-1   A
        CSRB.F  A
        CSRB.F  A               C[A] is now the X nibble coordinate.
        B=B+C   A               Add this offset into B[A]
        GOSUB   srR0            Get ABUFF
        C=C+B   A
        D0=C                    D0 points to one nibble up and left of ball.
        

        ?ST=0   1               Not going down?
        GOYES   dbBALL          Go draw ball,
        GOSUB   dbERLNE

dbBALL  GOSUB   D0DOWN          Move D0 down one line
        GOSUB   srR9            Direction
        ?CBIT=1 0               Going up?
        GOYES   dbB3U
        GOSUB   db3DOTS         Draw 3 dots (1st line of ball)
        GOTO    dbBTN
dbB3U   GOSUB   db3DIF          Draw these 3 lines if no erasing
                                is necessary.

dbBTN   GOSUB   D0DOWN
        GOSUB   srR9            Direction
        ?CBIT=1 0               Going up?
        GOYES   dbB5U
        GOSUB   db5DOTS         Draw 5 dots (2nd line of ball)
        GOTO    dbBT5N
dbB5U   GOSUB   db5DIF
        
dbBT5N  GOSUB   D0DOWN
        GOSUB   db5DOTS         Draw 5 dots (3rd line of ball)
        
        GOSUB   D0DOWN
        GOSUB   srR9
        ?CBIT=1 1
        GOYES   dbB5D
        GOSUB   db5DOTS         Draw 5 dots (4th line of ball)
        GOTO    dbBB5N
dbB5D   GOSUB   db5DIF
        
dbBB5N  GOSUB   D0DOWN
        GOSUB   srR9            Direction
        ?CBIT=1 1               Going down?
        GOYES   dbB3D
        GOSUB   db3DOTS         Draw 3 dots (5th line of ball)
        GOTO    dbBBN
dbB3D   GOSUB   db3DIF

dbBBN   GOSUB   D0DOWN
        ?ST=0   0               Not going up?
        GOYES   dbDEND          We're done,
        GOSUB   dbERLNE        

dbDEND  ST=0    0               Always clear direction flags.
        ST=0    1
        ST=0    2
        ST=0    3
        RTN

*****************************************************************
*
* dbERLNE, db3DOTS and db5DOTS use a series of reasonably complex
* set of masks for checking or drawing different pieces of the ball.
* First of all, 4 horizontal pixels are stored in one nibble.  If the
* ball is on an even nibble boundary, then it has a different bitmap
* pattern than if it were one pixel to the right, etc.  Because of this,
* there are 4 different bitmap patterns for the ball.  There are, therefore,
* 4 different patterns for the masks on the left of the ball, 4 more on
* the right, etc.  It's even worse than that!  On the left and right
* there is a mask for screen that is never erased, and a mask for screen
* that is sometimes erased (depending on direction) etc.
*
* I apologize that I will not comment this code in detail.  I had enough
* fun writing it and debugging it in the first place.
*
*****************************************************************
        
dbERLNE ST=0    4               Assume left
        GOSUB   srR9            Direction
        ?CBIT=0 3               Not going right?
        GOYES   dbE0
        ST=1    4
dbE0    GOSUB   srR13           Old X coord.
        ?CBIT=1 1
        GOYES   dbE2
        ?CBIT=1 0
        GOYES   dbE1
        ?ST=1   4               Moving right?
        GOYES   dbER1
        LCHEX   F1F             Position 0, left
        GOTO    dbEALL
dbER1   LCHEX   FF1             position 0, right
        GOTO    dbEALL
dbE1    ?ST=1   4               moving right?
        GOYES   dbER2
        LCHEX   E3F             Position 1, left
        GOTO    dbEALL
dbER2   LCHEX   FE3             Position 1, right
        GOTO    dbEALL
dbE2    ?CBIT=1 0
        GOYES   dbE3
        LCHEX   FC7             Position 2
        GOTO    dbEALL
dbE3    LCHEX   F8F
dbEALL  A=DAT0  3
        C=C&A   A
        DAT0=C  3
        RTN


db3DOTS A=DAT0  3               Get the 3 nibbles of screen
        B=A     A
        GOSUB   srR7            X coord.
        ?CBIT=1 1
        GOYES   db32
        ?CBIT=1 0
        GOYES   db31

        LCHEX   0E0             Ball bitmap
        D=C     A
        LCHEX   C07             Mask of screen we always keep
        A=0     A               Set the "sometimes" mask to blank
        ?ST=1   2               Moving left?
        GOYES   db30R
        LAHEX   300             Mask of (left) screen we sometimes keep
db30R   ?ST=0   3               Not moving right?
        GOYES   dbK3
        GOTO    dbALL
dbK3    LAHEX   018             Mask of (right) screen we sometimes keep
        GOTO    dbALL

db31    LCHEX   01C             ball
        D=C     A
        LCHEX   F80             always keep
        A=0     A
        ?ST=1   2
        GOYES   db31R
        LAHEX   060             left sometimes
db31R   ?ST=1   3
        GOYES   dbALL
        LAHEX   003             right sometimes
        GOTO    dbALL

db32    ?CBIT=1 0
        GOYES   db33
        LCHEX   038             ball
        D=C     A
        LCHEX   F01             always keep
        A=0     A
        ?ST=1   2
        GOYES   db32R
        LAHEX   0C0             left sometimes
db32R   ?ST=1   3
        GOYES   dbALL
        LAHEX   006             right sometimes
        GOTO    dbALL

db33    LCHEX   070             ball
        D=C     A
        LCHEX   E03             always keep
        A=0     A
        ?ST=1   2
        GOYES   db33R
        LAHEX   180             left sometimes

db33R   ?ST=1   3
        GOYES   dbALL
        LAHEX   00C             right sometimes

dbALL   C=C&B   A               Mask out screen we always keep
        A=A&B   A               Mask out screen we sometimes keep
        C=C!A   A               OR it in
        C=C!D   A               OR in the ball bitmap
        DAT0=C  3               write it to screen.
        RTN

db3DIF  A=DAT0  3
        GOSUB   srR7
        ?CBIT=1 1
        GOYES   db3D2
        ?CBIT=1 0
        GOYES   db3D1

        LCHEX   0E0             Ball bitmap
        GOTO    dbDALL
db3D1   LCHEX   01C             Ball
        GOTO    dbDALL
db3D2   ?CBIT=1 0
        GOYES   db3D3
        LCHEX   038             Ball
        GOTO    dbDALL
db3D3   LCHEX   070             Ball
dbDALL  C=C!A   A
        DAT0=C  3
        RTN
           

* Coding method for db5dots is very similar to db3dots.  Use the
* cryptic comments above to make inferences about code below. :)

db5DOTS A=DAT0  3
        B=A     A
        GOSUB   srR7
        ?CBIT=1 1
        GOYES   db52
        ?CBIT=1 0
        GOYES   db51

        LCHEX   1F0             ball
        D=C     A
        LCHEX   C07             always keep
        A=0     A
        ?ST=1   2
        GOYES   db50R
        LAHEX   200             left sometimes

db50R   ?ST=0   3
        GOYES   db50RK
        GOTO    dbALL
db50RK  LAHEX   008             right sometimes
        GOTO    dbALL

db51    LCHEX   03E
        D=C     A
        LCHEX   F80
        A=0     A
        ?ST=1   2
        GOYES   db51R
        LAHEX   040
db51R   ?ST=0   3
        GOYES   db51K
        GOTO    dbALL
db51K   LAHEX   001
        GOTO    dbALL

db52    ?CBIT=1 0
        GOYES   db53
        LCHEX   07C
        D=C     A
        LCHEX   F01
        A=0     A
        ?ST=1   2
        GOYES   db52R
        LAHEX   080
db52R   ?ST=0   3
        GOYES   dbK1
        GOTO    dbALL
dbK1    LAHEX   002
        GOTO    dbALL

db53    LCHEX   0F8
        D=C     A
        LCHEX   E03
        A=0     A
        ?ST=1   2
        GOYES   db53R
        LAHEX   100

db53R   ?ST=0   3
        GOYES   dbK2
        GOTO    dbALL
dbK2    LAHEX   004
        GOTO    dbALL

db5DIF  A=DAT0  3
        GOSUBL  srR7
        ?CBIT=1 1
        GOYES   db5D2
        ?CBIT=1 0
        GOYES   db5D1

        LCHEX   1F0             Ball bitmap
        GOTO    dbDALL
db5D1   LCHEX   03E             Ball
        GOTO    dbDALL
db5D2   ?CBIT=1 0
        GOYES   db5D3
        LCHEX   07C             Ball
        GOTO    dbDALL
db5D3   LCHEX   0F8             Ball
        GOTO    dbDALL

*****************************************************************
*
* Move the ball up one pixel (unless it is blocked.)
* If the ball "sees" an object above it, then it changes
* the direction from UP to DOWN, and calls the dbDOWN code.
* This makes it bounce.
*
* The pixels that the ball will "look at" depend on whether the
* ball is travelling up-left or up-right as follows:
*
*    UP-LEFT        UP-RIGHT
*
*     xxx              xxx      (mask #1)
*     x***            ***x      (mask #2)
*     *****          *****
*     *****          *****
*     *****          *****
*      ***            ***
*
* The asterisks represent the pixels of the ball shape.  The x characters
* represent the pixels that will be checked.  If any of the pixels in the
* x postions are on, the ball will bounce down.
*
*****************************************************************


dbUP    GOSUBL  srR8            Y coord
        GOSUBL   srS12           Store for dbLEFT or dbRIGHT
        C=C-1   A               One pixel above Y coord
        D=C     A               Save here for later
        LA(5)   34
        GOSBVL  =MUL#           B is the nibble offset to (Y-1) position.
        GOSUBL  srR7            X coord
        CSRB.F  A
        CSRB.F  A
        B=B+C   A               Add X coord nibble offset
        GOSUBL  srR0
        C=C+B   A
        D0=C

* In this next code segment, B[B] will contain mask #1, and
* A[B] will contain mask #2.  The masks are different depending
* on the position of the ball relative to a nibble boundary, and
* whether the ball is moving up-left or up-right.  This code
* will determine this, and shift the mask accordingly so that it
* will "look" at the correct area on the screen.
        
        LCHEX   07              mask #1
        B=C     B               Store mask #1 in B[B]
        LCHEX   1               mask #2
        A=C     B               Store mask #2 in A[B]

        GOSUBL  srR7            X coord
        ?CBIT=0 1               This checks to see if the ball is at least
                                2 pixels away from a nibble boundary.
                                If not, skip the following shifts, otherwise,
                                shift the masks.
        GOYES   dbUS1           skip
        B=B+B   B
        B=B+B   B
        A=A+A   B
        A=A+A   B
dbUS1   ?CBIT=0 0               This checks to see if the ball is at least
                                1 pixel away from a nibble boundary.
        GOYES   dbUS2           skip
        B=B+B   B
        A=A+A   B
dbUS2   GOSUBL  srR9            direction
        ?CBIT=1 2               Going left?
        GOYES   dbUS3
        B=B+B   B
        B=B+B   B
        ASL     B

dbUS3   C=DAT0  B               Read 2 nibbles of screen
        C=C&B   B               Mask #1 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbUBNCE         Yes, we must bounce
        GOSUB   D0DOWN          Move down one line
        C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbUBNCE         Yes, we must bounce
        
* If we get here, then there was no bounce.  We can move the ball up
* one pixel.
        
        C=D     A               Get the new Y coord.
        GOLONG  srS8            Store it.

dbUBNCE GOSUBL  srR9            Ball must bounce.  Get direction.
        GOSUBL  srS11           Store old DIR for dbLEFT and dbRIGHT
        CBIT=0  0               Not going up anymore.
        CBIT=1  1               Now going down.
        GOSUBL  srS9            store direction. (doesn't affect left/right)
        GOSUB   soWALL          make a WALL sound.

* Now we move the ball down instead.


*****************************************************************
*
* Move the ball down one pixel (unless it is blocked.)
* If the ball "sees" an object below it, then it changes
* the direction from DOWN to UP, and calls the dbUP code.
* This makes it bounce.
*
* The pixels that the ball will "look at" depend on whether the
* ball is travelling up-left or up-right as follows:
*
*   DOWN-LEFT      DOWN-RIGHT
*
*      ***            ***
*     *****          *****
*     *****          *****
*     *****          *****
*     x***            ***x      (mask #2)
*     xxx              xxx      (mask #1)
*
* The asterisks represent the pixels of the ball shape.  The x characters
* represent the pixels that will be checked.  If any of the pixels in the
* x postions are on, the ball will bounce up.
*
*****************************************************************

dbDOWN  GOSUBL  srR8            Y coord
        GOSUBL  srS12           Store for dbLEFT or dbRIGHT
        D=C     A               Save Y coord
        C=C+CON A,4             The bottom line of the ball
        LA(5)   34
        GOSBVL  =MUL#
        GOSUBL  srR7
        CSRB.F  A
        CSRB.F  A
        B=B+C   A               Add X coord nibble offset
        GOSUBL  srR0
        C=C+B   A
        D0=C

* B[B] will contain mask #1, and A[B] will contain mask #2.
* These masks are identical to those in dbUP and the location 
* is at the bottom of the ball, not the top.
* The comments for this code is very similar to dbUP.

        LCHEX   07              mask #1
        B=C     B
        LCHEX   1
        A=C     B

        GOSUBL  srR7            X coord
        ?CBIT=0 1
        GOYES   dbDS1
        B=B+B   B
        B=B+B   B
        A=A+A   B
        A=A+A   B
dbDS1   ?CBIT=0 0
        GOYES   dbDS2
        B=B+B   B
        A=A+A   B
dbDS2   GOSUBL  srR9            direction
        ?CBIT=1 2               Going left?
        GOYES   dbDS3
        B=B+B   B
        B=B+B   B
        ASL     B

dbDS3   C=DAT0  B               Read 2 nibbles of screen

* in dbDOWN, we use mask #2 first, because it is on top
* of mask #1 (unlike dbUP).

        C=C&A   B               Mask #2 of screen
        ?C#0    B               Is there something there?
        GOYES   dbDBNCE         Yes, we must bounce
        GOSUB   D0DOWN          Move down one line
        C=DAT0  B               Read 2 nibbles of screen
        C=C&B   B               Mask #1 of screen
        ?C#0    B               Is there something there?
        GOYES   dbDBNCE         Yes, we must bounce

        C=D     A               Get saved value
        C=C+1   A               Down one
        GOLONG  srS8            Save for real

dbDBNCE GOSUBL  srR9            Ball must bounce.  Get direction
        GOSUBL  srS11           Store old DIR for dbLEFT and dbRIGHT
        CBIT=0  1               Ball not going down anymore.
        CBIT=1  0               Now going up.
        GOSUBL  srS9            Store new direction.
        GOSUB   soWALL          Make wall sound
        GOTO    dbUP            Move ball up one pixel.


*****************************************************************
*
* Move the ball left one pixel (unless it is blocked.)
* If the ball "sees" an object to the left, then it changes
* the direction from LEFT to RIGHT, and calls the dbRIGHT code.
* This makes it bounce.
*
* The pixels that the ball will "look at" depend on whether the
* ball is travelling up-left or down-left as follows:
*
*    UP-LEFT       DOWN-LEFT
*
*    xx***            ***
*    x*****          *****
*    x*****         x*****
*     *****         x*****
*      ***          xx***
*
* The asterisks represent the pixels of the ball shape.  The x characters
* represent the pixels that will be checked.  If any of the pixels in the
* x postions are on, the ball will bounce right.
*
*****************************************************************

dbLEFT  GOSUBL  srR12           Old Y coord
        A=C     A               Put it into A[A]

* We need to point to the correct mask starting line.
* If the ball was moving up, we point to the top line,
* if the ball was moving down, we add 2 to the Old Y coord.
        
        GOSUBL  srR11           Old Direction
        ?CBIT=1 0               Was going up?
        GOYES   dbLNXT          Yes, do nothing.
        A=A+CON A,2             Point to the correct screen line.
        
dbLNXT  LC(5)   34
        GOSBVL  =MUL#           B[A] = A[A] * C[A]
        GOSUBL  srR7            X coord
        GOSUBL  srS13           Save it here for dbDRAW
        C=C-1   A               one pixel to the left.
        ?C#0    A
        GOYES   dbLCHK          Did we hit the left of screen?
        GOTO    msMISS          Yes... the left player just missed the ball.
dbLCHK  D=C     A               Store new X coord here for later.
        CSRB.F  A
        CSRB.F  A
        B=B+C   A
        GOSUBL  srR0
        C=C+B   A
        D0=C                    D0 points to the nibble that holds the
                                pixel to the left of the X coord.

        LCHEX   03              mask #1
        B=C     B
        LCHEX   1               mask #2
        A=C     B

        C=D     A               Get (X-1)
        ?CBIT=0 1
        GOYES   dbLS1
        B=B+B   B
        B=B+B   B
        A=A+A   B
        A=A+A   B

dbLS1   ?CBIT=0 0
        GOYES   dbLS2
        B=B+B   B
        A=A+A   B

dbLS2   GOSUBL  srR11           Direction
        ?CBIT=1 1               Going down?
        GOYES   dbLDOWN

        C=DAT0  B               Read 2 nibbles of screen
        C=C&B   B               Mask #1 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbLBNCE         Yes, we must bounce
        
        GOSUBL   D0DOWN
        C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbLBNCE         Yes, we must bounce
        
        GOSUBL  D0DOWN
        C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbLBNCE         Yes, we must bounce
        GOTO    dbLOK           No bounce!

dbLDOWN C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbLBNCE         Yes, we must bounce
        
        GOSUBL  D0DOWN
        C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbLBNCE         Yes, we must bounce
        
        GOSUBL  D0DOWN
        C=DAT0  B               Read 2 nibbles of screen
        C=C&B   B               Mask #1 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbLBNCE         Yes, we must bounce

dbLOK   C=D     A               Ball didn't bounce,
        GOLONG  srS7            Store new X coord.

dbLBNCE GOSUBL  srR9            Ball must bounce, get direction.
        CBIT=0  2               Not going left anymore.
        CBIT=1  3               Now going right.
        GOSUBL  srS9            Store new direction
        GOSUB   soPDL           Make a paddle sound.
                                Note that we don't actually know that the
                                ball has hit a paddle.  If it ever hits
                                an obstacle that causes it to bounce to the
                                right, it makes this sound.

*****************************************************************
*
* Move the ball right one pixel (unless it is blocked.)
* If the ball "sees" an object to the right, then it changes
* the direction from RIGHT to LEFT, and calls the dbLEFT code.
* This makes it bounce.
*
* The pixels that the ball will "look at" depend on whether the
* ball is travelling up-right or down-right as follows:
*
*    UP-RIGHT      DOWN-RIGHT
*
*      ***xx          ***
*     *****x         *****
*     *****x         *****x
*     *****          *****x
*      ***            ***xx
*
* The asterisks represent the pixels of the ball shape.  The x characters
* represent the pixels that will be checked.  If any of the pixels in the
* x postions are on, the ball will bounce left.
*
*****************************************************************

dbRIGHT GOSUBL  srR12           Old Y coord
        A=C     A               Put it into A[A]

* We need to point to the correct mask starting line.
* If the ball was moving up, we point to the top line,
* if the ball was moving down, we add 2 to the Old Y coord.
        
        GOSUBL  srR11           Old Direction
        ?CBIT=1 0               Was going up?
        GOYES   dbRNXT          Yes, do nothing.
        A=A+CON A,2             Point to the correct screen line.
        
dbRNXT  LC(5)   34
        GOSBVL  =MUL#
        GOSUBL  srR7            X coord.
        GOSUBL  srS13           Save it here for dbDRAW
        C=C+1   A
        LA(5)   128
        ?C<A    A
        GOYES   dbRCHK
        GOTO    msMISS          Ball hit right of screen, right player
                                just missed the ball!

dbRCHK  D=C     A               Store (X+1)
        C=C+CON A,3             Move to pixel postion we want to "see"
        CSRB.F  A
        CSRB.F  A
        B=B+C   A
        GOSUBL  srR0
        C=C+B   A
        D0=C

        LCHEX   03              mask #1
        B=C     B
        LCHEX   2               mask #2
        A=C     B

        GOSUBL  srR7            X coord.  Even though this X coord
                                is 4 pixels to the left of the screen
                                area we are looking at, it works OK,
                                because bits 0 and 1 are the same, and
                                that's all we care about here.
        ?CBIT=0 1
        GOYES   dbRS1
        B=B+B   B
        B=B+B   B
        A=A+A   B
        A=A+A   B

dbRS1   ?CBIT=0 0
        GOYES   dbRS2
        B=B+B   B
        A=A+A   B

dbRS2   GOSUBL  srR11           Old Direction
        ?CBIT=1 1               Going down?
        GOYES   dbRDOWN

        C=DAT0  B               Read 2 nibbles of screen
        C=C&B   B               Mask #1 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbRBNCE         Yes, we must bounce
        
        GOSUBL  D0DOWN
        C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbRBNCE         Yes, we must bounce
        
        GOSUBL  D0DOWN
        C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbRBNCE         Yes, we must bounce
        GOTO    dbROK           No bounce!

dbRDOWN C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbRBNCE         Yes, we must bounce
        
        GOSUBL  D0DOWN
        C=DAT0  B               Read 2 nibbles of screen
        C=C&A   B               Mask #2 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbRBNCE         Yes, we must bounce
        
        GOSUBL  D0DOWN
        C=DAT0  B               Read 2 nibbles of screen
        C=C&B   B               Mask #1 of screen.
        ?C#0    B               Is there something there?
        GOYES   dbRBNCE         Yes, we must bounce


dbROK   C=D     A               Ball didn't bounce.
        GOLONG  srS7            Store new X coord.

dbRBNCE GOSUBL  srR9            Ball must bounce, get direction.
        CBIT=0  3               not going right anymore.
        CBIT=1  2               now going left.
        GOSUBL  srS9            store new direction.
        GOSUB   soPDL           Make a paddle sound
                                Note that we don't actually know that the
                                ball has hit a paddle.  If it ever hits
                                an obstacle that causes it to bounce to the
                                left, it makes this sound.
        
        GOTO    dbLEFT          Go move left one pixel.


msMISS  C=RSTK                  POP return address off of stack.
                                There was a GOSUB dbRIGHT, or dbLEFT
                                to get here, and we're about to exit
                                to RPL.  Thus, we must pop that return
                                address from the stack, because we
                                aren't going to return.
        LA(2)   255             Sound when someone misses the ball
        LC(2)   30
        GOSUB   Tone
        GOLONG  kyEXIT


*****************************************************************
*
* dlDELAY will delay C[A] ticks.  (1 tick is 1/8192 seconds)
* Beware that =GetTimChk alters A, B, C, D, P, D1 and CARRY and
* uses 3 RSTK levels.
*
* Using a delay that is based on the internal clock is the best
* method.  This way the delays are the same amount of time on an
* SX (slower processor) as they are on a GX (faster processor).
*
*****************************************************************


dlDELAY R4=C                    Delay n ticks (in C[W])
        GOSBVL  =GetTimChk      (#012EEh) This routine returns the 13 
                                nibble system time into C[W].
        R2=C                    Current time into R2
dlLP    GOSBVL  =GetTimChk      time
        A=R2                    start time
        P=      12
        C=C-A   WP              elapsed time
        A=R4                    n
        ?A>=C   WP              delay some more?
        GOYES   dlLP
        P=      0
        RTN

*****************************************************************
*
* dlINIT will capture the 13 nibble CPU time and put it into R3[W].
* Beware that =GetTimChk alters A, B, C, D, P, D1 and CARRY and
* uses 3 RSTK levels.
*
*****************************************************************


dlINIT  GOSBVL  =GetTimChk
        R3=C
        RTN

*****************************************************************
*
* dlCHECK will check if enough time has elapsed since the ball
* was last moved.  If the ball delay is set to 256, then the ball
* will be moved once every 256 ticks (about).  1 tick is 1/8192 seconds,
* so this means that the ball will be moved about 32 times every second.
*
* dlINIT is called to "start" the timer, and dlCHECK is called many times
* to see if time is up yet.  When the time is up, ST[5] is cleared to
* signal that the ball should be moved again.
*
*****************************************************************

dlCHECK C=0     W               Clear out C[W]
        GOSUBL  srR10           Get ball delay (in ticks)
        R4=C                    Put it into R4 (This is R4[W])
        GOSBVL  =GetTimChk      Get current CPU time into C[W]
        A=R3                    Get starting CPU time
        P=      12
        C=C-A   WP              C[W] contains elapsed time (since dlINIT)
        A=R4
        ?A>=C   WP              Time's up?
        GOYES   dlMORE          No?  Do more.
        P=      0               Set P back to 0
        ST=0    5               We can move the ball now.
        RTN 
dlMORE  P=      0               Set P back to 0
        ST=1    5               We can't move the ball yet.
        RTN


soWALL  LA(2)   170             Sound played when a wall is hit
        LC(2)   6
        GOTO    Tone

soPDL   LA(2)   90              Sound played when a paddle is hit
        LC(2)   12

Tone    ?ST=1   11              Is sound off?
        RTNYES                  Return if sound is off.
                                Note that this is not totally correct.
                                When the sound is off you should delay
                                for exactly the amount of time the sound
                                would have taken.  This will cause the
                                game to "feel" the same even when the sound
                                is off.  Each time the ball bounces on a
                                wall, there is a very small delay while the
                                sound is played.  If this delay is gone when
                                the sound is off, the game doesn't "feel"
                                the same.  For simplicity, this is not done.

        D=C     B               Copy intermediate delay to D[B]
ToneLp  LCHEX   800             Set bit 11
        OUT=C   B               Click speaker ON
        C=A     B               Copy tone value
Dec1    C=C-1   B               Delay
        GONC    Dec1
        C=0     A               Clear bit 11
        OUT=C                   Click speaker OFF
        C=A     B               Copy tone value
Dec2    C=C-1   B               Delay
        GONC    Dec2
        D=D-1   B               Decrement tone length counter
        GONC    ToneLp          Loop
        RTN


ENDCODE


  TURNMENUON
  RECLAIMDISP

  2DROP

;


