*****************************************************************
*****************************************************************
**
*+ Name: RVTest - Test the RVIEW Subroutine
**
** Author: Douglas R. Cannon
**
** Abstract:   Set up a known CPU context to test the
**             functionality of the RVIEW utility.
**
** Stack:                 --->
**
**     Date     Prog             Modification
**   --------   ----   ------------------------------------------
**   10/20/94   DC     Standard Header
**===============================================================


ASSEMBLE
        NIBASC  /HPHP48-L/

RPL

::
  TURNMENUOFF

CODE
        
        GOSBVL  =SAVPTR
 
        LCHEX   0123456789ABCDEF        Give D[W] a known value
        D=C     W
        LCHEX   1111222233334444        Give B[W] a known value
        B=C     W
        LAHEX   010101010100FACE        Give A[W] a known value
        LC(5)   1
        RSTK=C                  Push 00001 to RSTK
        LCHEX   ABCDE
        RSTK=C                  Push ABCDE to RSTK
        C=0     A
        C=C-1   A               Set the CARRY
                
        P=      7               Set P

        GOSUB   RVIEW           Call RVIEW

        C=RSTK                  Get the ABCDE off RSTK
        C=RSTK                  Get 00001 off RSTK

        GOVLNG =GETPTRLOOP      Back to RPL



*****************************************************************
*****************************************************************
**
*+ Name: RVIEW - Register Viewer
**
** Author: Douglas R. Cannon
**
** Abstract:   Save the current CPU context, display that context
**             for viewing, including other debugging utilities.
**
** Entry:      None
**
** Exit:       None
**
**     Date     Prog             Modification
**   --------   ----   ------------------------------------------
**   10/20/94   DC     Standard Header
**===============================================================

*******************************
*
* Global assignments
*
* Status Regs:
*
*  0: (0):GX, (1):SX
*  1: Key repeater
*  2: (0):SCR1, (1):SCR2
*  3: 
*  4: (0):Menu1, (1):Menu2 (for scr1)
*  4:
*  5:
*  6:
*  7:
*  8:
*  9:
* 10:
* 11:
*
* R0: ABUFF bitmap addr.
* R1: storage addr.
* R2: Used in D16, D5
* R3: Used in RSHOW
* R4: Used in D16
*
********************************

TIMERCTRL.1     EQU     #0012E
ADISP.GX        EQU     #806D5
ADISP.SX        EQU     #70556

*****************************************************************
*
* RVIEW is designed as an assembly language debugging
* tool.  You should be able to place a GOSUBL RVIEW
* call anywhere in your assembly code, and when execution
* gets to that point, RVIEW will display the current
* CPU context.  When you exit RVIEW, your code will
* continue to execute as if there was no interruption.
*
* One main disadvantage to RVIEW is that there will need
* to be at least 3 levels of RSTK free just before the
* call to RVIEW is made.  These levels are used for:
* 
* 1: GOSUBL RVIEW pushes return address to RSTK.
* 2: C[A] is pushed to RSTK to preserve it.
* 3: A GOSUB STORE is called to get the address of the
*    internal RAM storage space used by RVIEW.
*
* The RAM storage shown below is used for storing the current
* CPU context.  It is generally very bad practise to use RAM
* space inside your code object for any kind of data storage.
* This is one case where it can be acceptable.  I can't use RAM
* space anywhere else, because the user may want to use it in
* his/her application.  The only space I know the user won't possibly
* be using is inside the RVIEW code area.  Since RVIEW is intended
* as a temporary debugging aid, this "cardinal sin" can be pardoned.
* FOR THIS REASON, RVIEW MUST NEVER BE EXECUTED FROM A WRITE-PROTECTED
* RAM CARD.
* 
*****************************************************************


RVIEW   RSTK=C
        GOSUBL STORE            This puts the address of the first
                                NIBHEX onto RSTK.  It is a convenient
                                way of locating internal data.

        NIBHEX  0000000000       D0, changable D0               (0)
        NIBHEX  0000000000       D1, changable D1               (10)
        NIBHEX  00000            MEM                            (20)
        NIBHEX  0000000000000000 A                              (25)
        NIBHEX  0000000000000000 B                              (41)
        NIBHEX  0000000000000000 C                              (57)
        NIBHEX  0000000000000000 D                              (73)
        NIBHEX  0000000000000000 R0                             (89)
        NIBHEX  0000000000000000 R1                             (105)
        NIBHEX  0000000000000000 R2                             (121)
        NIBHEX  0000000000000000 R3                             (137)
        NIBHEX  0000000000000000 R4                             (153)
        NIBHEX  0                P                              (169)
        NIBHEX  0                CARRY                          (170)
        NIBHEX  0000             ST                             (171)
        NIBHEX  0                MODE                           (175)
        NIBHEX  000000000000000  RSTK                           (176)
        NIBHEX  000000000000000  RSTK
        NIBHEX  0000000000       RSTK (40 nibbles total)

sD0     EQU     0
sD0CH   EQU     5
sD1     EQU     10
sD1CH   EQU     15
sMEM    EQU     20
sA      EQU     25
sB      EQU     41
sC      EQU     57
sD      EQU     73
sR0     EQU     89
sR1     EQU     105
SR2     EQU     121
SR3     EQU     137
SR4     EQU     153
sP      EQU     169
sCRRY   EQU     170
sST     EQU     171
sMODE   EQU     175
sRSTK   EQU     176
sTOT    EQU     216

* RVIEW uses ABUFF for its display.  The user may be using ABUFF        
* in his/her application.  For this reason, RVIEW will save the
* current ABUFF before modifying it.  It will save it here, internally.
* The bitmap area of ABUFF is 1088 bytes.  Each line of ABUFF has
* 34 nibbles, and there are 64 lines.  There are 64 NIBHEX lines
* below, each containing 34 nibbles.
*        
* The address of this ABUFF storage area is never stored.
* Since it is only used twice (once at the start, and once at
* the end) it's address is always calculated.  (R0[A] + sTOT)
        
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000      8
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000      16
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000      24
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000      32
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000      40
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000      48
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000      56
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000
        NIBHEX  0000000000000000000000000000000000      64


*****************************************************************
*
* This STORE routine will store the CPU context into the RAM
* area shown above.  The storing method is pretty straightforward
* except at the very first.  It's a little tricky to store the
* entire CPU context, including the CARRY.  The CARRY must be
* stored before any registers are incremented or decremented.
*
*****************************************************************


STORE   C=RSTK
        CD0EX
        DAT0=C A                Store D0.  This value won't change.

        C=P    0                P in C[0]
        P=     1                Set P to 1 so that the next two LCHEX
                                lines will put the value into C[1],
                                not C[0].
       
        LCHEX  1                SET CARRY in C[1]
        GOC    CSET
        LCHEX  0                CLEAR CARRY in C[1]
CSET    D0=D0+ 5
        DAT0=C 2                Save P, CARRY (1 if set, 0 if clear)
        P=     0                P needs to be 0.
       
* The Carry had to be saved before incrementing
* D0, otherwise it would be cleared.  Later,
* This area is used for the changeable D0, and
* The Carry is stored in the proper place.
* Also, the Carry couldn't be saved unless P was
* known.  For this reason, I saved P too.

        D0=D0+  5               Incr. to the D1 area.

* Because RVIEW is self-modifying, it can't be executed from a
* write-protected RAM card.  I can use C[A] to see if the user is 
* executing RVIEW from a write-protected RAM card.  If they are, 
* I will immediately exit, even though D0, CARRY, and P will be lost.
* This will prevent a fatal crash.  If the user's code depends on
* D0, it may crash anyway... but then that's their fault!
*
* First I will write a 0 and check if it's a 1.  Then I'll write a
* 1 and check if it's a 0.  If either case is true, then they're executing
* from a write-protected RAM card.

        C=0     A               Put a 0 into C[A]
        DAT0=C  1               write it out to the storage area
        C=DAT0  1               read it back
        ?CBIT=1 0               is bit 0 set?
        GOYES   EXIT            If so, I must exit

        CBIT=1  0               Put a 1 into C[1]
        DAT0=C  1               write it out to the storage area
        C=DAT0  1               read it back
        ?CBIT=1 0               is bit 0 set?
        GOYES   STD1            If so, it's OK... continue

EXIT    C=RSTK                  Get C[A] back off of RSTK
        RTN                     Emergency Exit: D0, CARRY, and P are lost.
        
STD1    CD1EX
        DAT0=C  A               Store D1.  This value won't change.
        D0=D0+  5
        DAT0=C  A               Store D1.  This value can change.
        D0=D0+  5               

* I'll assume that they're on a GX at first, but later this
* value will be changed to #70000h if it's discovered that they're
* running this code on an SX.  The start of the User-RAM is
* #70000h on an SX and #80000h on a GX.

        LCHEX   80000           MEM starting area
        DAT0=C  A
        D0=D0+  5
                 
        DAT0=A  W               Store A register
        D0=D0+  16

        A=B     W               Store B register
        DAT0=A  W
        D0=D0+  16

        C=RSTK                  Store C register
        DAT0=C  W
        D0=D0+  16

        C=D     W               Store D register
        DAT0=C  W
        D0=D0+  16

        C=R0                    R0
        DAT0=C W
        D0=D0+ 16

        C=R1                    R1
        DAT0=C W
        D0=D0+ 16

        C=R2                    R2
        DAT0=C W
        D0=D0+ 16

        C=R3                    R3
        DAT0=C W
        D0=D0+ 16

        C=R4                    R4
        DAT0=C W
        D0=D0+ 16

        D0=D0+  2               Skip past P and CARRY.

        C=0     A               Clear out A field in C
        C=ST                    Get bits 0-11 of ST into C[X]
       
* Now, get bits 12-15 of ST.  It's trickier.
        ?ST=0   12
        GOYES   C13
        CBIT=1  12
C13     ?ST=0   13
        GOYES   C14
        CBIT=1  13
C14     ?ST=0   14
        GOYES   C15
        CBIT=1  14
C15     ?ST=0   15
        GOYES   SAVEST
        CBIT=1  15
SAVEST  DAT0=C  4               Save ST
        D0=D0+  4

        LCHEX   9
        C=C+1   P
        LCHEX   1
        GOC     DECMD
        C=0     A
DECMD   DAT0=C  1               Save MODE (1 = DEC, 0 = HEX)
        
* Up to this point, it hasn't mattered if the CPU were in
* HEXMODE or DECMODE.  From now on I need to assume that the
* CPU is in HEXMODE, so I set it here.
        
        SETHEX                  I need HEXMODE
        D0=D0+  1

        LA(2)   8-1             RSTK  (40 nibbles)
RSTKSV  C=RSTK
        DAT0=C  A
        D0=D0+  5
        A=A-1   B
        GONC    RSTKSV

**************

* Save storage addr

        AD0EX
        LC(5)   sTOT
        C=A-C   A
        R1=C.F  A               This is the address of the Storage

        D0=C
        LA(5)   sP              offset to the P, CARRY storage
        C=C+A   A
        D1=C                    Point D1 to correct storage area
        A=DAT0  A               Read D0
        D0=D0+  5               Point to the temp P, CARRY.
        C=DAT0  2               Read correct P, CARRY
        DAT1=C  2               Write P, CARRY in correct area.
        DAT0=A  A               Write the changeable D0.
       
* Get screen addr

* First point D1 to the MEM address area.  the MEM window could
* point anywhere, but I will default it to the start of the User-RAM
* area.  On an SX, this is #70000h, and on a GX it is #80000h.

        A=R1.F  A               Get Storage address
        LC(5)   (sMEM)+4
        C=C+A   A               Point to MEM address (high-order nibble)
        D1=C

        D0=(5)  (=INHARDROM?)+14     (#1A2E8)
        C=DAT0  1               7=SX, 8=GX
        DAT1=C  1               Write out the value to MEM address
        ?CBIT=1 3
        GOYES   GX
        
        D0=(5)  =ADISP.SX       ADISP addr for SX
        ST=1    0
        GOTO    BOTH

GX      D0=(5)  =ADISP.GX       ADISP addr for GX
        ST=0    0

BOTH    A=DAT0  A
        LC(5)   20
        C=C+A   A
        R0=C.F  A

* I haven't modified ABUFF yet, but I have its address.  I will
* preserve the current ABUFF first.
        
        C=R0.F  A
        D0=C                    put ABUFF bitmap addr into D0
        A=R1.F  A               Get Storage address
        LC(5)   sTOT            get offset to ABUFF storage
        C=C+A   A
        D1=C                    put storage addr into D1

        LC(2)   136-1           I save 8 bytes at a time: 8*136 = 1088
        B=C     B
        C=0     W
SVABUFF A=DAT0  W               read 8 bytes
        DAT1=A  W               store 8 bytes
        DAT0=C  W               clear ABUFF at the same time
        D0=D0+  16
        D1=D1+  16
        B=B-1   B               decrement counter
        GONC    SVABUFF         loop

********************
* MAIN INITS

        ST=0    2               Screen1 is showing (scr1)
        ST=0    4               Menu1 is showing (on scr1)
        ST=0    15              I need interrupts off
SHOW1   GOSUB   SCR1            Show screen 1
        GOLONG  GETKEY          Wait for a key



*****************************
* Screen #1
*

* This method of displaying graphics on the screen
* is pretty straightforward.  The NIBHEX statements
* are a bitmap of the graphic I want to display.
* By executing a GOSUB, RSTK contains the address
* of the bitmap data.  This is the address of where
* execution would continue if a RTN were encountered.
* However, I never RTN, I get the address off of RSTK
* instead.
*
* Once the address of the bitmap data is known, a quick
* loop is done to store the data into the ABUFF RAM area,
* which displays the graphic.

SCR1    GOSUB  S1G1

        NIBHEX  0200052007000520
        NIBHEX  0500000003000520
        NIBHEX  0300052003000000
        NIBHEX  0600012001000120
        NIBHEX  0600000003000520
        NIBHEX  0500052003000000
        NIBHEX  0000000000000000
        NIBHEX  0000000013203552
        NIBHEX  7550355213200000
        NIBHEX  0320053205200522
        NIBHEX  0370000004400C62
        NIBHEX  045004420440

* Display A:, B:, C:, D:,>D0:, D1:, M:
  
S1G1    C=RSTK
        D1=C                    Put the graphic data address into D1
        C=R0.F  A               
        D0=C                    Put the ABUFF bitmap address into D0
                                Note that no offset is added to the
                                ABUFF address.  This means that the graphic
                                will be displayed in the very upper-left
                                corner of the HP48 screen.
        LC(2)   47-1            This graphic is 47 pixels high
S1G1LP  A=DAT1  4               This graphic is 4 nibbles wide
        DAT0=A  4               Store the 4 nibbles into ABUFF
        D1=D1+  4               Add 4 to our graphic data pointer
        D0=D0+  16              There are 34 nibbles in one horizontal
        D0=D0+  16              line on the HP48 screen.  (Even though
        D0=D0+  2               the last 5 pixels of each horizontal
                                line are not visible).  To move our ABUFF
                                pointer down 1 line, I add 34 to it.
        C=C-1   B               Decrement our height counter
        GONC    S1G1LP          Do more, if necessary

        GOSUB   S1G2            Display the next graphic using the same
                                method.

        NIBHEX  360512310112
        NIBHEX  160000060012
        NIBHEX  010012060000
        NIBHEX  030052030012
        NIBHEX  010
       
S1G2    C=RSTK                  Show PC:, C:, P:
        D1=C
        A=R0.F  A
        LC(5)   21
        C=C+A   A
        D0=C
        LC(2)   17-1
S1G2LP  A=DAT1  3
        DAT0=A  3
        D1=D1+  3
        D0=D0+  16
        D0=D0+  16
        D0=D0+  2
        C=C-1   B
        GONC    S1G2LP

        GOSUB   s1MENU          Draw the current menu
        
        GOSUB   S1G4

        NIBHEX  202000202000202

S1G4    C=RSTK                  Draw the colons after 5 nib addresses
        D1=C
        A=R0.F  A
        LC(5)   1063
        C=C+A   A
        D0=C
        LC(2)   15-1
S1G4LP  A=DAT1  1
        DAT0=A  1
        D1=D1+  1
        D0=D0+  16
        D0=D0+  16
        D0=D0+  2
        C=C-1   B
        GONC    S1G4LP


* Show the register contents for scr1

       
        LC(5)   1024            Screen offset to D0 area
        D=C     A
        LC(5)   5               RAM data offset (D0)
        GOSUBL  dsRAM           Display D0 area

        LC(5)   1228            Screen offset to D1 area
        D=C     A
        LC(5)   15              RAM data offset (D1)
        GOSUBL  dsRAM           Display D1 area

        LC(5)   1432            Screen offset to MEM area
        D=C     A
        LC(5)   20              RAM data offset (MEM)
        GOSUBL  dsRAM           Display MEM area

        C=R1.F  A
        D0=C    
        D0=D0+  15
       
        LC(5)   3               Show A register
        D=C     A
        D0=D0+  10              D0 now points to A reg
        C=DAT0  W
        B=C     W
        GOSUBL  D16

        LC(5)   207             Show B register
        D=C     A
        D0=D0+  16
        C=DAT0  W
        B=C     W
        GOSUBL  D16

        LC(5)   411             Show C register
        D=C     A
        D0=D0+  16
        C=DAT0  W
        B=C     W
        GOSUBL  D16

        LC(5)   615             Show D register
        D=C     A
        D0=D0+  16
        C=DAT0  W
        B=C     W
        GOSUBL  D16

* Note that because the user must call RVIEW with a GOSUB, the
* top level of RSTK contained the return address, or in other
* words, the Program Counter.
        
        LC(5)   24              Show the PC
        D=C     A
        LC(5)   103             sRSTK - sD
        AD0EX                   
        C=C+A   A
        D0=C                    Point to RSTK, top level (PC)
        C=DAT0  A
        B=C     A
        GOSUBL  D5

        LC(5)   228             Show CARRY
        D=C     A
        D0=D0-  6               (sRSTK - sCRRY)
        C=DAT0  1        
        B=C     A
        GOSUBL  DNUM            Display 1 number, D0 is lost

        LC(5)   432             Show P
        D=C     A
        A=R1.F  A
        LC(5)   sP              offset to P
        C=C+A   A
        D0=C                    D0 points to P
        C=DAT0  1
        B=C     A
        GOSUBL  DNUM            Display 1 number, D0 is lost

        A=R1.F  A               Show MODE next
        LC(5)   sMODE           offset to MODE
        C=C+A   A
        D0=C
        C=0     A               Zero out C[A]
        C=DAT0  1               (1 = DEC, 0 = HEX)
        ?C=0    A
        GOYES   HEXDAT

        GOSUB   DSPMD
        NIBHEX  3761DDC1        data for DECMODE
        NIBHEX  511B5550
        NIBHEX  571555D1
        NIBHEX  51115550
        NIBHEX  3761DDC1

HEXDAT  GOSUB   DSPMD
        NIBHEX  5751DDC1        data for HEXMODE
        NIBHEX  515B5550
        NIBHEX  772555D1
        NIBHEX  51515550
        NIBHEX  5751DDC1

* Depending on where the GOSUB was done, the bitmap data
* will show DECMODE or HEXMODE.  This display loop is the
* same, regardless.

DSPMD   C=RSTK
        D1=C
        A=R0.F  A
        LC(5)   634
        C=C+A   A
        D0=C
        LC(2)   5-1
DMLOOP  A=DAT1  8
        DAT0=A  8
        D1=D1+  8
        D0=D0+  16
        D0=D0+  16
        D0=D0+  2
        C=C-1   B
        GONC    DMLOOP

        RTN

* This here will just draw the current menu
* for screen one.  It does not modify ST[3].

s1MENU  ?ST=0   4               Am I looking at menu1?
        GOYES   s1MENU1
        GOTO    s1MENU2
        
* Toggle between menu1 and menu2 for scr1
* If ST=0 3, then I'm looking at menu1
* If ST=1 3, then I'm looking at menu2

s1MTOG  ?ST=1   4               Am I looking at menu2?
        GOYES   s1SET1
        GOTO    s1SET2
s1SET1  ST=0    4               Set it to be menu1
s1MENU1 GOSUB   s1DRMNU

* SCR1, MENU1
        NIBHEX  FFFFFDFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70
        NIBHEX  BB8C8DB9997FF7EFD
        NIBHEX  FF9F7FF7CFDFF1F70
        NIBHEX  39AAED55557FFFEFD
        NIBHEX  FBBF7FF7FFDFBDF70
        NIBHEX  BAAC8D15597F7CEFD
        NIBHEX  F1BF7F74CFDF11F70
        NIBHEX  BBAAED55557FFFEFD
        NIBHEX  FBBF7FFFDFDFB7F70
        NIBHEX  BB8A8D59957FF7CFD
        NIBHEX  FF1F7FF7CFDFF1F70
        NIBHEX  FFFFFDFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70

s1SET2  ST=1    4               Set it to be menu2
s1MENU2 GOSUB   s1DRMNU

* SCR1, MENU2
        NIBHEX  FFFFFDFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70
        NIBHEX  BB8C8DB9997F7EEED
        NIBHEX  F9BB7F9BBBD7EEE60
        NIBHEX  39AAED55557FF65DD
        NIBHEX  BB557FB555DE65550
        NIBHEX  BAAC8D155977C65DD
        NIBHEX  1B5571B5555C65550
        NIBHEX  BBAAED55557FF65DD
        NIBHEX  BB557FB555DE65550
        NIBHEX  BB8A8D59957F7CEED
        NIBHEX  F1BB7F1BBBD7CEE60
        NIBHEX  FFFFFDFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70

s1DRMNU C=RSTK                  Draw the menu bar (scr1)
        D1=C
        A=R0.F  A
        LC(5)   1938
        C=C+A   A
        D0=C
        LC(2)   17-1
s1DMLP  A=DAT1  14
        DAT0=A  14
        D1=D1+  14
        D0=D0+  14
        C=C-1   B
        GONC    s1DMLP
        RTN

*****************************
* Screen #2
*

SCR2    GOSUB  S2G1

        NIBHEX  320552350552520
        NIBHEX  000320532320522
        NIBHEX  570000370542370
        NIBHEX  512570000370542
        NIBHEX  370542570000350
        NIBHEX  552370542540

S2G1    C=RSTK                  Display R0:, R1:, R2:, R3:, R4:
        D1=C
        C=R0.F A
        D0=C
        LC(2)  29-1
S2G1LP  A=DAT1 3
        DAT0=A 3
        D1=D1+ 3
        D0=D0+ 16
        D0=D0+ 16
        D0=D0+ 2
        C=C-1  B
        GONC   S2G1LP

        GOSUB  S2G2

        NIBHEX  360512310112160

S2G2    C=RSTK                  Display PC:
        D1=C
        A=R0.F A
        LC(5)  225
        C=C+A  A
        D0=C
        LC(2)  5-1
S2G2LP  A=DAT1 3
        DAT0=A 3
        D1=D1+ 3
        D0=D0+ 16
        D0=D0+ 16
        D0=D0+ 2
        C=C-1  B
        GONC   S2G2LP

        GOSUB  S2G3

        NIBHEX  36750
        NIBHEX  51252
        NIBHEX  32230
        NIBHEX  54252
        NIBHEX  53250

S2G3    C=RSTK                  Display RSTK:
        D1=C
        A=R0.F A
        LC(5)  24
        C=C+A  A
        D0=C
        LC(2)  5-1
S2G3LP  A=DAT1 5
        DAT0=A 5
        D1=D1+ 5
        D0=D0+ 16
        D0=D0+ 16
        D0=D0+ 2
        C=C-1  B
        GONC   S2G3LP

        GOSUB  S2G4

        NIBHEX  FFFFFDFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70
        NIBHEX  BB8CFDFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70
        NIBHEX  39AA8DFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70
        NIBHEX  BAAC8DFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70
        NIBHEX  BBAA8DFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70
        NIBHEX  BB8AFDFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70
        NIBHEX  FFFFFDFFFF7FFFFFD
        NIBHEX  FFFF7FFFFFDFFFF70

S2G4    C=RSTK                  Draw the menu bar (scr2)
        D1=C
        A=R0.F A
        LC(5)  1938
        C=C+A  A
        D0=C
        LC(2)  17-1
S2G4LP  A=DAT1 14
        DAT0=A 14
        D1=D1+ 14
        D0=D0+ 14
        C=C-1  B
        GONC   S2G4LP

* Show the register contents for scr2

        A=R1.F  A
        LC(5)   sR0             Offset to R0
        C=C+A   A               C is pointing to R0 storage.
        D0=C

        LC(5)   3               Show R0 register
        D=C     A
        C=DAT0  W
        B=C     W
        GOSUBL  D16

        LC(5)   207             Show R1 register
        D=C     A
        D0=D0+  16              Point to R1
        C=DAT0  W
        B=C     W
        GOSUBL  D16

        LC(5)   411             Show R2 register
        D=C     A
        D0=D0+  16              Point to R2
        C=DAT0  W
        B=C     W
        GOSUBL  D16

        LC(5)   615             Show R3 register
        D=C     A
        D0=D0+  16              Point to R3
        C=DAT0  W
        B=C     W
        GOSUBL  D16

        LC(5)   819             Show R4 register
        D=C     A
        D0=D0+  16              Point to R4
        C=DAT0  W
        B=C     W
        GOSUBL  D16

        LC(5)   228
        D=C     A
        D0=D0+  16
        D0=D0+  7               (sRSTK-sP) Point to RSTK storage
        LC(2)   8-1
        R3=C.F  B
 
RSHOW   C=DAT0 A
        B=C    A
        GOSUBL D5
        D0=D0+ 5                Point to the next RSTK level
        LC(5)  (34*6)-5         6 lines
        D=D+C  A                move down 6 lines

        C=R3.F B                temp counter
        C=C-1  B                one less level to display
        R3=C.F B
        GONC   RSHOW            if more, show them.

* Show the Status bits
        
        GOSUB   s2STGR
        
        NIBHEX  670             The ST: GROB
        NIBHEX  121
        NIBHEX  220
        NIBHEX  421
        NIBHEX  320
        
s2STGR  C=RSTK
        D1=C
        LC(5)   1632
        A=R0.F  A
        C=C+A   A
        D0=C
        LC(2)  5-1
s2SGLP  A=DAT1 3
        DAT0=A 3
        D1=D1+ 3
        D0=D0+ 16
        D0=D0+ 16
        D0=D0+ 2
        C=C-1  B
        GONC   s2SGLP

        LC(5)   1159            offset to the 111111
        D=C     A               offset in D
        LCHEX   11111           a buncha ones
        B=C     A
        GOSUBL  D5              Display them
        GOSUBL  DNUM            display one more 1. (D was changed)

        LC(5)   1363            offset to the 5432109876543210
        D=C     A               
        LCHEX   5432109876543210
        B=C     W
        GOSUBL  D16

        LC(5)   1567
        A=R0.F  A               Get screen bitmap addr
        C=C+A   A
        D0=C
        LCHEX   7777777777777777
        DAT0=C  W               Draw a dotted line

        LC(5)   sST             Get the ST offset
        A=R1.F  A               RAM offset
        C=C+A   A
        D0=C
        A=DAT0  4               Read the 16 bits
        LC(2)   16-1            Do this loop 16 times
        B=C     B
s2STLP  CSL     W               Shift C left 1 nibble
        ?ABIT=0 15              highest bit is zero?
        GOYES   s2NEXT
        C=C+1   A               Make the lower nibble 1
s2NEXT  A=A+A   A               Shift A left 1 bit
        B=B-1   B
        GONC    s2STLP

* Now, C[W] contains the 16 status bits, spread out into 16 nibbles

        B=C     W
        LC(5)   1635            Offset
        D=C     A
        GOSUBL  D16             Display the 16 bits

        RTN


SCRTOG  ?ST=0  2                Am I looking at scr1?
        GOYES  DOSCR2
        ST=0   2                Now, I'll be looking at scr1

        GOSUB  ERASE            erase ABUFF
        GOLONG SCR1             display scr1  (auto RTN)

DOSCR2  ST=1   2                Now, I'll be looking at scr2
        GOSUB  ERASE
        GOTO   SCR2             display scr2  (auto RTN)

* Erase ABUFF, but not the menu bar.  This will erase the
* top 56 lines of ABUFF.  In this application, the entire menu
* bar is replaced by SCR1 and SCR2, so there is no need to
* erase it.
* (56 * 34) / 16 = 119, so there are 119 sets of 16 nibbles
* to be erased.

ERASE   C=R0.F  A               Get ABUFF bitmap addr.
        D0=C
        LC(2)   119-1           119 times...
        A=0     W               blank out A[W]
ERLP    DAT0=A  W               erase 16 nibbles
        D0=D0+  16              move to the next 16 nibbles
        C=C-1   B               one less in counter
        GONC    ERLP            done yet?
        RTN


* Check for keys

GETKEY  ST=0    1

* Shut off the busy annunciator to indicate that
* RVIEW is waiting for a keypress.

BUSYOFF D0=(5)  (=ANNCTRL)+1
        C=DAT0  1
        CBIT=0  0
        DAT0=C  1
              
* Enter a light sleep mode.  This conserves battery power
* while waiting for a keypress because the CPU has been
* Shut down.  

LITE    D1=(5)  =TIMERCTRL.1
        LC(1)   4
        DAT1=C  1
        D1=(2)  =TIMER1
        LC(1)   7               Wait (7+1)/16 seconds between wakeups
        DAT1=C  1
        LC(3)   (=allkeys)+1024         all keys, including ON
                                        The user can press any key to
                                        wakeup the CPU.
        OUT=C

WAIT    SHUTDN                  Enter light sleep mode
        
        LC(3)   (=allkeys)+1024
        OUT=C
        GOSBVL  =CINRTN         Find out which key woke us up
        LAHEX   0803F           In mask for all keys (including ON)
        A=A&C   A
        ?A#0    A               A key was pressed?
        GOYES   BUSYON          Go turn on the Busy annunciator
        
        D1=(2)  =TIMERCTRL.1
        C=DAT1  3
        ?CBIT=0 3               No TIMER1 WAKEUP?        
        GOYES   WAIT
        GOTO    LITE
                      
* Turn on the busy annunciator while deciding which key was pressed.

BUSYON  D0=(5)  (=ANNCTRL)+1
        C=DAT0  1
        CBIT=1  0
        DAT0=C  1

CHA     LCHEX   002             Check the A key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00010
        A=A&C   A
        ?A=0    A               Not the A key?
        GOYES   DOB?            Go see if the B key should be checked.

        GOSUB   SCRTOG          Toggle screens
        GOTO    STOPKEY         Get ready to get another key

DOB?    ?ST=0   2               Am I on scr1?
        GOYES   CHB             Yes?  OK, check other keys
        GOTO    CHBS            No? Skip to backspace key

CHB     LCHEX   100             Check the B key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00010
        A=A&C   A
        ?A=0    A
        GOYES   CHC

        GOSUB   adADDR          Prompt the user for an address
        GOTO    STOPKEY         Get ready to get another key
        
CHC     LCHEX   100             Check the C key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00008
        A=A&C   A
        ?A#0    A               Did they press C?
        GOYES   CMENU           Go process it

        LCHEX   040             Check for the right arrow key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00001
        A=A&C   A
        ?A=0    A               They're not pressing this one?
        GOYES   CHD             Go check the D key

* The C key will subtract #1h or #100h from the current MEM address,
* depending on which menu is currently showing.
        
CMENU   ?ST=0   4               Menu1?
        GOYES   s1L1            Go to Screen1, Left 1
        LCHEX   00100
        GOTO    s1LEFT          Move left #100h nibbles
s1L1    LCHEX   00001
        GOTO    s1LEFT          Move left #1h nibble
                                                      
* If you're careful, you can check for more than one key at
* a time, as is done below.  This works when checking for the
* D key and the right arrow key.  However, I couldn't do this
* for the C key and the left arrow key above.  If I had, then
* I would have used an OUT mask of 140, and an IN mask of 00009.
* This gets the two keys just fine, but notice that it also
* gets the F key, which is OUT: 100, IN: 00001

CHD     LCHEX   140             Check the D or left arrow key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00004
        A=A&C   A
        ?A=0    A
        GOYES   CHE

* The D key will add #1h or #100h to the current MEM address,
* depending on which menu is currently showing.
        
        ?ST=0   4               Menu1?
        GOYES   s1R1
        LCHEX   00100
        GOTO    s1RIGHT         Move right #100h nibbles
s1R1    LCHEX   00001
        GOTO    s1RIGHT         Move right #1h nibble

CHE     LCHEX   100             Check the E key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00002
        A=A&C   A
        ?A=0    A
        GOYES   CHF

* The E key will subtract #5h or #1000h from the current MEM address,
* depending on which menu is currently showing.
        
        ?ST=0   4               Menu1?
        GOYES   s1L5
        LCHEX   01000
        GOTO    s1LEFT          Move left #1000h nibbles
s1L5    LCHEX   00005           Move left #5h nibbles

* Move the current RAM "window" left C[A] nibbles

s1LEFT  B=C     A
        GOSUB   gcGETCD         Get offsets C[A], D[A]
        A=R1.F  A               Get RAM data address
        A=A+C   A
        D0=A
        A=DAT0  A               Read the mem addr
        A=A-B   A               Decrement by B[A]
        DAT0=A  A               Store it
        GOSUBL  dsRAM
        GOTO    PSEKEY          PSEKEY will pause, then autorepeat
                                if the user is holding the key down.

CHF     LCHEX   100             Check the F key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00001
        A=A&C   A
        ?A=0    A
        GOYES   CHNXT            

* The F key will add #5h or #1000h to the current MEM address,
* depending on which menu is currently showing.
        
        ?ST=0   4               Menu1?
        GOYES   s1R5
        LCHEX   01000
        GOTO    s1RIGHT         Move right #1000h nibbles
s1R5    LCHEX   00005           Move right #5h nibbles


* Move D0 or D1 "window" right C[A] nibbles

s1RIGHT B=C     A
        GOSUB   gcGETCD         Get offsets C[A], D[A]
        A=R1.F  A               Get RAM data address
        A=A+C   A
        D0=A
        A=DAT0  A               Read the mem addr
        A=A+B   A               Increment by #100h
        DAT0=A  A               Store it
        GOSUB   dsRAM
        GOTO    PSEKEY          PSEKEY will pause, then autorepeat
                                if the user is holding the key down.
        
CHNXT   LCHEX   080             Check the NXT key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00001
        A=A&C   A
        ?A=0    A
        GOYES   CHDOWN

        GOSUBL  s1MTOG          Toggle the menu
        GOTO    STOPKEY

CHDOWN  LCHEX   040             Check the down arrow key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00002
        A=A&C   A
        ?A=0    A
        GOYES   CHUP

        GOSUB   tmTOGM          Move the menu arrow down one
        GOTO    STOPKEY

CHUP    LCHEX   080             Check the up arrow key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00002
        A=A&C   A
        ?A=0    A
        GOYES   CHBS

* I need to move the MEM arrow UP one.  The easiest way is to just 
* call tmTOGM twice.  Since speed is not a concern for this function,
* this is acceptable.  tmTOGM is pretty fast anyway.  Doing it this 
* way makes moving the arrow up twice as slow as moving it down.  
* I dare you to see if you can even tell the difference.

        GOSUB   tmTOGM
        GOSUB   tmTOGM
        GOTO    STOPKEY
        

CHBS    LCHEX   410             Check for [<-] or [ON]
        OUT=C                   
        GOSBVL  =CINRTN
        LAHEX   08001
        A=A&C   A
        ?A#0    A               One was pressed?
        GOYES   BSSTOP          Go wait until they stop
        GOTO    GETKEY          Otherwise, go get another key

* One interesting thing about assembly language: it's extremely fast!
* When the average user just presses a key normally, from assembly
* language it appears that they've held it down for a very long time.
* Each time a key is pressed, there has to be a loop coded to wait
* until the user stops pressing the key.  This particular loop will
* wait until the user stops pressing the Backspace or ON key.

BSSTOP  LCHEX   410
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   08001
        A=A&C   A
        ?A#0    A               Still holding it down?
        GOYES   BSSTOP          Continue looping
        GOTO    RESTOR          Otherwise, start exiting RVIEW

* This loop will wait until they stop pressing any of the defined
* keys other than Backspace or ON.  During this loop, the key's
* function has already been performed, I'm just waiting for the
* user to let go of the key so I can go check for the next keypress.

STOPKEY LCHEX   1FF             
        OUT=C                
        GOSBVL  =CINRTN         
        LAHEX   0003F
        A=A&C   A
        ?A#0    A               They're still holding it down?
        GOYES   STOPKEY         Continue looping
        GOTO    GETKEY          It's OK to go check for the next key

* If the user is holding one of the keys that will move the
* RAM window (screen1, keys C, D, E, and F), then I want to
* pause for a short while and then autorepeat the function.

PSEKEY  C=0     A
        LCHEX   150             Pause this much while repeating
        B=C     A
        ?ST=1   1               0 = first time, 1 = subsequent loops
        GOYES   PSEWAIT
        LCHEX   2A00            Pause this much the first time
        B=C     A

PSEWAIT LCHEX   140             Check if they're pressing one still
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   0000F
        A=A&C   A
        ?A=0    A               No key being pressed?
        GOYES   PSEOUT          Get out of the loop
        B=B-1   A               Decrement the pause amount
        GONC    PSEWAIT         Continue waiting, if necessary
        ST=1    1               This is no longer the first time
        GOTO    CHC             Go see which one they're pressing
PSEOUT  GOTO    GETKEY          Go get a new key


*****************************************************************
*
* RESTOR will Restore the original CPU context.  When the RTN
* is executed, CPU execution will resume to the user's code
* as if it had never been interrupted.
*
*****************************************************************

RESTOR  A=R1.F  A
        LC(5)   sTOT
        A=A+C   A
        D1=A                    Get the address of ABUFF storage
        C=R0.F  A
        D0=C                    put ABUFF addr into D0

        LC(2)   136-1           I save 8 bytes at a time: 8*136 = 1088
RSABUFF A=DAT1  W               read 8 bytes
        DAT0=A  W               store 8 bytes
        D0=D0+  16
        D1=D1+  16
        C=C-1   B               decrement counter
        GONC    RSABUFF         loop
        
        A=R1.F  A               Get storage addr
        LC(5)   sTOT
        A=A+C   A               Move to the end value
        D0=A
        
        LA(2)   8-1             Restore RSTK
RSTKRS  D0=D0-  5
        C=DAT0  A
        RSTK=C
        A=A-1   B
        GONC    RSTKRS

        D0=D0-  1               Point to MODE nibble
        C=DAT0  1               (1 = DEC, 0 = HEX)
        ?CBIT=0 0               Do I need HEXMODE?
        GOYES   STRS            Do nothing, I'm already in HEXMODE.
        SETDEC                  Put CPU back in DECMODE.  It's OK
                                To be in DECMODE from here to the end
                                because instructions which increment or
                                decrement D0 are always performed in
                                HEXMODE.

STRS    D0=D0-  4               Point to the Status bits
        C=DAT0  4               Read them into C.

* Bits 12-14 are not modified and need not be restored.

        ST=C                    Restore bits 0-11 from C[X]
        ?CBIT=0 15              I set ST 15 to 0.  Check if it was already.
        GOYES   STDONE          Yes? Do nothing.
        ST=1    15              Otherwise, set it.

STDONE  A=R1.F  A               Get storage address.
        D1=A
        D1=D1+  5               Point to changeable D0.
        D0=D0-  2               Point to P, CARRY.
        C=DAT0  2               Get P, CARRY.
        DAT1=C  2               Store them here for use later.
                                P and CARRY must be restored last.

        D0=D0-  16              Restore R4
        C=DAT0  W
        R4=C

        D0=D0-  16              Restore R3
        C=DAT0  W
        R3=C

        D0=D0-  16              Restore R2
        C=DAT0  W
        R2=C

        D0=D0-  16              Restore R1
        C=DAT0  W
        R1=C

        D0=D0-  16              Restore R0
        C=DAT0  W
        R0=C

        D0=D0-  16              Restore D
        C=DAT0  W
        D=C     W

        D0=D0-  16              Restore C
        C=DAT0  W
        RSTK=C                  Save C[A] to RSTK.

        D0=D0-  16              Restore B
        A=DAT0  W
        B=A     W

        D0=D0-  16              Restore A
        A=DAT0  W

        D0=D0-  15              Skip past MEM area, point to D1
        C=DAT0  A
        D1=C                    Restore D1

        D0=D0-  5               Point to P, CARRY
        C=DAT0  2               Get P, CARRY
        P=C     0               Restore P.  From this point on it's OK
                                to have P be any value
        
        D0=D0-  5

* I had to do the last decrement to D0 prior to executing this
* next code.  Anytime a register is incremented or decremented and
* there is no overflow, CARRY is cleared.  I am about to restore
* CARRY to it's original value, so it must not be modified after
* this is done, yet C[A] and D0 are not yet restored.  I can still
* restore them without modifying the CARRY.
*
* The next two lines of code will set or clear the CARRY depending
* on what the value of the CARRY was when RVIEW was called.  The
* CARRY is set if a test is true, and the CARRY is cleared if a test
* is false.  The value of the CARRY is in CBIT[4], thus by executing
* the line ?CBIT=1 4, if the test is true and the CARRY should be set,
* then the CARRY gets set.  Otherwise, if CBIT[4] is 0, and the CARRY
* should be cleared, then the test will be false, and the CARRY will
* be cleared.  The GOYES is not needed, so it is set to simply go to
* the next line.

        ?CBIT=1 4               Restore the CARRY
        GOYES   D0RS            GOYES does nothing

D0RS    C=DAT0  A               Get D0
        D0=C                    Restore D0

        C=RSTK                  Restore C[A]

        RTN                     Return to the User's code

************************
* General Subroutines



* tmTOGM will simply move the arrow from
* D0 to D1 to M.  It does nothing more.
* When I increment one of these areas, it will
* Check to see where the arrow is, then act accordingly.
*
* Uses D0, D1, A[A], C[A], B[B], and D[A]

tmTOGM  A=R0.F  A               * Get bitmap addr
        LC(5)   1020            * Offset to D0 arrow
        A=A+C   A               * A contains D0 arrow tip address
        D0=A                    * Put this into D0
        C=DAT0  1
        ?CBIT=0 0               * If clear, I didn't find the arrow
        GOYES   tmCHD1
        LC(5)   1224            * Set to the D1 area
        GOTO    tmDRAW          * erase area, draw new arrow

tmCHD1  LC(5)   204             * skip 6 lines
        A=A+C   A
        D0=A
        C=DAT0  1
        ?CBIT=0 0
        GOYES   tmCHM
        LC(5)   1428            * Set to the MEM area
        GOTO    tmDRAW

tmCHM   LC(5)   1020            * Set to the D0 area
tmDRAW  D=C     A               * Put offset into A
        LC(2)   17-1            * Clear 17 lines
        B=C     B
        A=R0.F  A               * Get bitmap addr
        LC(5)   1020            * Offset to top of arrow area
        C=C+A   A
        D0=C
        C=0     A               * Clear out C[A]
tmCLOOP DAT0=C  1               * Blank 1 nibble
        D0=D0+  16
        D0=D0+  16
        D0=D0+  2               * Move to next line
        B=B-1   B
        GONC    tmCLOOP         * Loop until done

        C=R0.F  A               * Get bitmap addr
        C=C+D   A               * Add offset to new arrow spot
        D0=C                    * Put this into D0

        GOSUB   tmDATA          * put addr of data onto RSTK
        NIBHEX  13731
tmDATA  C=RSTK
        D1=C                    * D1 points to data
        LC(2)   5-1             * 5 lines in one arrow
tmDLOOP A=DAT1  1               * Get 1 nibble of arrow
        DAT0=A  1               * Draw it
        D1=D1+  1               * increment data pointer
        D0=D0+  16
        D0=D0+  16
        D0=D0+  2               * Move down 1 line on screen
        C=C-1   B
        GONC    tmDLOOP

        RTN

* adADDR will check which mem address has the pointer,
* and it will prompt you for a new address.  It will
* show a cursor and wait for keypresses 0-9, A-F.
* ON will cancel, and BS will backspace.
*
* Uses A[A], B[A], C[A], D[A], D0, D1, R2[A], R3[A], R4[A]
*
* R2[A] : digit counter
* R3[A] : address they type
* R4[A] : cursor offset

adADDR  GOSUB   gcGETCD         Get screen offset into D[A]
        C=R0.F  A               Get bitmap addr
        C=C+D   A
        D0=C
        A=0     A
        LC(2)   5-1             Erase 5 lines
adERASE DAT0=A  A               Erase 5 nibbles
        D0=D0+  16
        D0=D0+  16
        D0=D0+  2               Move down 1 line
        C=C-1   B
        GONC    adERASE         erase the next line

        C=0     A
        R4=C.F  A               set cursor offset to 0
        LC(1)   5               5 digits in an address
        R2=C.F  A               set digit counter to 5
        GOTO    adSTOP          They just pushed [A] to get here
        
* This is the key loop
adKEY   D0=(5)  (=ANNCTRL)+1    Turn off BUSY annunciator
        C=DAT0  1
        CBIT=0  0
        DAT0=C  1
              
adLITE  D1=(5)  =TIMERCTRL.1
        LC(1)   4
        DAT1=C  1
        D1=(2)  =TIMER1
        LC(1)   0               (0+1)/16 seconds delay
        DAT1=C  1
        LC(3)   (=allkeys)+1024 allkeys, including ON
        OUT=C

adWAIT  SHUTDN
        
        LC(3)   (=allkeys)+1024
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   0803F
        A=A&C   A
        ?A#0    A
        GOYES   adBON           They hit a key... exit loop
        
        D1=(2)  =TIMERCTRL.1
        C=DAT1  3
        ?CBIT=0 3               No TIMER1 WAKEUP?        
        GOYES   adWAIT
        GOSUB   adCURS          Move the cursor
        GOTO    adLITE

adBON   D0=(5)  (=ANNCTRL)+1    Turn on the buzy annunciator
        C=DAT0  1
        CBIT=1  0
        DAT0=C  1

        GOSUB   adKEYS          Check keys (0-9) and (A-F)

* This data is for each key (0-9) and (A-F).        
* There are 5 nibbles per key used as follows:
* The first 3 nibbles are the OUT mask (backwards).
* The next 2 nibbles are the IN mask (backwards).

        NIBHEX  10080           0
        NIBHEX  20080           1
        NIBHEX  20040           2
        NIBHEX  20020           3
        NIBHEX  40080           4
        NIBHEX  40040           5
        NIBHEX  40020           6
        NIBHEX  80080           7
        NIBHEX  80040           8
        NIBHEX  80020           9
        NIBHEX  20001           A
        NIBHEX  00101           B
        NIBHEX  00180           C
        NIBHEX  00140           D
        NIBHEX  00120           E
        NIBHEX  00110           F
        
adKEYS  C=RSTK                  Get address of above data
        D1=C

* B[A] is the key counter.  It will be equal to which key
* I'm currently testing.  (00000 - 0000F)  It is then used
* in adHITK when a key is pressed.

        B=0     A               Start at 0, work up to F
adNEXT  C=DAT1  3               get the OUT mask
        D1=D1+  3               increment data pointer
        A=0     A               clear A[A] for IN mask
        A=DAT1  2               get the IN mask
        D1=D1+  2               increment data pointer
        OUT=C
        GOSBVL  =CINRTN
        A=A&C   A
        ?A=0    A               was this key not pressed?
        GOYES   adINC           do next key
        GOTO    adHITK          otherwise, go process it
adINC   B=B+1   A               increment the key counter
        LC(2)   15              Last key to test
        ?B<=C   B               Are there more to test?
        GOYES   adNEXT          Yes? go do it

* Now check if the user has pressed ON or BS.

adCHON  LCHEX   400             Check if they've pressed [ON]
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   08000
        A=A&C   A
        ?A=0    A               Did they not press [ON]?
        GOYES   adCHBS          Go check if they pressed BS

adONLP  LCHEX   400             Wait until they stop pressing ON
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   08000
        A=A&C   A
        ?A#0    A               Still pressing it?
        GOYES   adONLP          Then go check again
        GOSUB   gcGETCD
        GOTO    dsRAM           show old address, auto RTN

adCHBS  LCHEX   010             Check the Backspace key
        OUT=C
        GOSBVL  =CINRTN
        LAHEX   00001
        A=A&C   A
        ?A#0    A               Did they press the BS key?
        GOYES   adBS
        GOTO    adSTOP          The key they pressed is not defined        

adBS    A=R2.F  A               Get the digit number
        LC(5)   5               I started at 5
        ?A#C    A               Not at the start?
        GOYES   adBSOK          
        GOTO    adSTOP          at start, go wait for a key
adBSOK  A=A+1   A               One more digit to get now
        R2=A.F  A
        A=R3.F  A               Get address
        ASR     A               Shift right
        R3=A.F  A
        D=D-1   A               Back up one on screen

* Now I erase 2 nibbles at the new cursor position.
* I have to erase the digit here anyway, and the old
* cursor next to it.

        C=R0.F  A               Get bitmap addr
        C=C+D   A               Add offset
        D0=C
        A=0     A
        LC(2)   5-1             Erase 5 lines
adBSER  DAT0=A  2               Erase 2 nibbles
        D0=D0+  16
        D0=D0+  16
        D0=D0+  2               Move down 1 line
        C=C-1   B
        GONC    adBSER          erase the next line
        GOTO    adSTOP


adHITK  GOSUB   DNUM            Display the key they pressed  (in B[1])
        D=D+1   A               increment screen offset
        C=B     A               Get the digit back
        A=R3.F  A               Get the number
        ASL     A
        C=C+A   A               Add on the new digit
        R3=C.F  A               Store it
        A=R2.F  A
        A=A-1   A
        ?A#0    A               More digits still?
        GOYES   adMORE
        GOSUB   gcGETCD         Get the offset into storage
        A=R1.F  A
        A=A+C   A               A points to address storage
        D0=A
        A=R3.F  A               Get the new address
        DAT0=A  A               Store it
        GOTO    dsRAM           display it, auto RTN

adMORE  R2=A.F  A

adSTOP  LCHEX   1FF             Wait here until they stop
        OUT=C                   Pressing a key
        GOSBVL  =CINRTN
        LAHEX   0003F
        A=A&C   A
        ?A#0    A
        GOYES   adSTOP
        GOTO    adKEY

* This routine moves the cool looking cursor.
* D[A] must be set to the offset,
* Uses A[A], C[A], D0, R4[A]

adCURS  C=R4.F  A               Get current cursor offset
        C=C+D   A               add screen offset
        A=R0.F  A               Get screen bitmap
        C=C+A   A 
        D0=C
        C=0     A
        DAT0=C  1               Erase old cursor
        A=R4.F  A               Get old pos
        LC(5)   34              one line
        A=A+C   A
        LC(5)   170             Past the max
        ?A=C    A               Did I go too far?
        GOYES   adZERO
        R4=A.F  A
        GOTO    adDRAW          Go draw the new cursor
adZERO  A=0     A
        R4=A.F  A               Back to zero

adDRAW  C=A     A               Get cursor offset
        C=C+D   A
        A=R0.F  A
        C=C+A   A
        D0=C
        LCHEX   F
        DAT0=C  1               Draw one line
        RTN


* gcGETCD will get the RAM area offset into C[A],
* and the screen offset into D[A].
* Uses A[A], C[A], D[A], D0.

gcGETCD A=R0.F  A               Get bitmap addr
        LC(5)   1020            Offset to D0 arrow
        A=A+C   A               A contains D0 arrow tip address
        D0=A                    Put this into D0
        C=DAT0  1
        ?CBIT=0 0               If clear, I didn't find the arrow
        GOYES   gcCHD1
        LC(5)   1024            Offset to screen addr area
        D=C     A               Store here for use in dsRAM
        LC(5)   sD0CH           Set to the changeable D0 area
        RTN

gcCHD1  LC(5)   204             skip 6 lines
        A=A+C   A
        D0=A
        C=DAT0  1
        ?CBIT=0 0
        GOYES   gcCHM
        LC(5)   1228            Offset to screen addr area
        D=C     A               Store here for use in dsRAM
        LC(5)   sD1CH           Set to the changeable D1 area
        RTN

gcCHM   LC(5)   1432            Offset to screen addr area
        D=C     A               Store here for use in dsRAM
        LC(5)   sMEM            Set to the MEM area
        RTN

* dsRAM will display an address and 16 nibbles of what it
* points to.  C[A] must contain sD0CH, sD1CH or sMEM for the D0, D1
* or MEM address offsets.  D[A] should contain 1024, 1228, or
* 1432 as screen offsets of where to display the data.
*
* For example, if I want to update the D0 RAM area, I would
* do this:  LC(5) 1024, D=C A, LC(5) sD0CH, GOSUB dsRAM.

dsRAM   B=C     A               Save C[A]
        CD0EX
        RSTK=C                  Preserve D0
        A=R1.F  A               Get RAM data addr
        C=B     A               Get offset
        C=C+A   A               Add offset
        D0=C                    
        C=DAT0  A               Read the address
        B=C     A
        GOSUB   D5              Remember, D[A] is already set,
                                so display addr on screen

        D=D+1   A               Increment past the colon
        C=DAT0  A               Read the addr again
        D0=C            
        C=DAT0  W               Read what it points to
        P=      0
dsREV   B=C     P
        BSLC
        BSLC
        P=P+1
        GONC    dsREV
        BSRC
        GOSUB   D16ST           Display it
        C=RSTK
        D0=C                    Restore D0.
        RTN


**************

* Offset in D[A], number in B[W]

D16     CD0EX
        R2=C.F  A               preserve D0
D16ST   LC(2)   16-1
D16LP   BSLC
        R4=C.F  B
        GOSUB   DNUM
        D=D+1   A
        C=R4.F  B
        C=C-1   B
        GONC    D16LP
        C=R2.F  A
        D0=C
        RTN

* Offset in D[A], number in B[A]

D5      CD0EX
        R2=C.F  A               Preserve D0
        LC(2)   5-1
D5LP    BSRC
        C=C-1   B
        GONC    D5LP
        LC(2)   5-1
        GOTO    D16LP

**************

* Display a number (0-F) anywhere on the screen
* at a nibble boundary.
*
* D[A] contains the nibble offset
* B[1] contains the digit.
* A[A], C[A], D0, and D1 are changed.
* 

DNUM    GOSUB   DNSTRT

        NIBHEX  25552           0  (font bitmap)
        NIBHEX  23227           1
        NIBHEX  74717           2
        NIBHEX  74747           3
        NIBHEX  55744           4
        NIBHEX  71747           5
        NIBHEX  71757           6
        NIBHEX  74222           7
        NIBHEX  75757           8
        NIBHEX  75747           9
        NIBHEX  75755           A
        NIBHEX  35353           B
        NIBHEX  61116           C
        NIBHEX  35553           D
        NIBHEX  71317           E
        NIBHEX  71311           F

DNSTRT  C=R0.F  A
        C=C+D   A
        D0=C
        LC(5)   15
        C=C&B   A
        A=C     A
        A=A+A   A
        A=A+A   A
        A=A+C   A
        C=RSTK
        C=C+A   A
        D1=C
        LC(2)   5-1

DNLOOP  A=DAT1  1
        DAT0=A  1
        D1=D1+  1
        D0=D0+  16
        D0=D0+  16
        D0=D0+  2
        C=C-1   B
        GONC    DNLOOP

        RTN

***************


ENDCODE

;
