/*--------------------------------------------------------------------------
 Routines to build and print error messages.
 Includes:  ErrStr, ProgTitle, lose_title(), lose(), warn(), and PrintErr().
--------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <varargs.h>
#include <errno.h>
#include "lose.h"

/*LINTLIBRARY*/
#ifndef lint
static char Sccs_id[] = "@(#)lose.c from 1.16 5/31/87 (C) SRS";
#endif

#define MAX_STR_LEN 200
#define MAX_ERR_STR_LEN 1000

/* Private variables */
static char title[MAX_STR_LEN] = "program_name";
static char errorString[MAX_ERR_STR_LEN] = "";

/* Public variables */
char *ProgTitle = title;
char *ErrStr = errorString;

/*--------------------------------------------------------------------------
 Store the title of the program, initialize errno to zero.
--------------------------------------------------------------------------*/
void lose_title(char *name)
{
    errno = 0;
    ProgTitle[MAX_STR_LEN-1] = '\0';
    (void) strncpy(ProgTitle, name, MAX_STR_LEN-1);
}


/*--------------------------------------------------------------------------
 Build an error message, then maybe print it to stderr and maybe exit,
 as indicated by errMode.
--------------------------------------------------------------------------*/

static void _print_error(int errMode, char *format, va_list ap)
{
    extern int sys_nerr;
    extern char *sys_errlist[];

    /* Print the program name to the buffer */
    (void) sprintf(ErrStr, "%s: ", ProgTitle);

    /* Append the message to the buffer */
    vsprintf(ErrStr + strlen(ErrStr), format, ap);

    /* Append the system error message, if any */
    if (errno > 0 && errno < sys_nerr) {
    (void) strcat(ErrStr, "-- ");
    (void) strcat(ErrStr, sys_errlist[errno]);
    }

    /* Print the message to the console and/or exit, as indicated by errMode */
    switch(errMode) {
    case ERR_FATAL: 
    (void) fprintf(stderr, "%s\n", ErrStr);
    exit(1);
    break;
    case ERR_WARN: 
    (void) fprintf(stderr, "%s\n", ErrStr);
    break;
    case ERR_NOTE: 
    break;
    default:
    (void) fprintf(stderr, "%s\n", ErrStr);
    lose("illegal call to PrintErr with error mode %d\n", errMode);
    break;
    }
}

/*--------------------------------------------------------------------------
 Lint definitions to make lint shut up...
 you may safely omit this section if you don't use lint.
--------------------------------------------------------------------------*/
#ifdef lint
    /*ARGSUSED*/
    /*VARARGS1*/            /* Only check first argument */
    void warn(message) char *message;   /* First arg must be char *  */
    { message[0]=0; }           /* Dummy function body       */

    /*ARGSUSED*/
    /*VARARGS1*/            /* Only check first argument */
    void lose(message) char *message;   /* First arg must be char *  */
    { message[0]=0; }           /* Dummy function body       */

    /*ARGSUSED*/
    /*VARARGS1*/            /* Only check first argument */
    void Epitaph(message) char *message;/* First arg must be char *  */
    { message[0]=0; }           /* Dummy function body       */

    /*ARGSUSED*/
    /*VARARGS2*/            /* Check first 2 arguments   */
    void PrintErr(errMode, message)
    int errMode;            /* First arg must be int     */
    char *message;          /* Second arg must be char * */
    { message[0]=(char)errMode; }   /* Dummy function body       */
#else

/*--------------------------------------------------------------------------
 Various methods of calling _print_error():

 For nonfatal errors:
    warn(format, ...)    -> _print_error(ERR_WARN,  format, ...)

 For fatal I/O errors (this one is the most useful):
    lose(format, ...)    -> _print_error(ERR_FATAL, format, ...)

 For fatal non-I/O errors:
    Epitaph(format, ...) -> errno=0, _print_error(ERR_FATAL, format, ...)

 For nonfatal errors when you don't want message printed to stderr:
    PrintErr(errMode, format, ...) -> _print_error(errMode, format, ...)
--------------------------------------------------------------------------*/
/*VARARGS2*/
void PrintErr(errMode,format,va_alist)
	int errMode;                  
	char *format;
	va_dcl
{
    va_list ap;
    va_start(ap);
    _print_error(errMode, format, ap);
}

/*VARARGS1*/
void
warn(format, va_alist)
    char *format;
    va_dcl
{
    va_list ap;
    va_start(ap);
    _print_error(ERR_WARN, format, ap);
}

/*VARARGS1*/
void
lose(format, va_alist)
    char *format;
    va_dcl
{
    va_list ap;
    va_start(ap);
    _print_error(ERR_FATAL, format, ap);
}

/*VARARGS1*/
void
Epitaph(format, va_alist)
    char *format;
    va_dcl
{
    va_list ap;
    va_start(ap);
    errno = 0;
    _print_error(ERR_FATAL, format, ap);
}

#endif
